/**
 * @file generic_client.c
 *
 * @brief  Workflow example : Generic client for workflow submission
 *
 * @author  Abdelkader AMAR (Abdelkader.Amar@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

#include <string.h>
#ifndef __WIN32__
#include <unistd.h>
#include <sys/time.h>
#include <libgen.h>
#else
#include <Winsock2.h>
#include <windows.h>
#include <sys/timeb.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>
#include <time.h>
/*#include <libgen.h> basename() */
#include "DIET_client.h"


/* argv[1]: client config file path
   argv[2]: path of the worflow description file */
#ifdef __WIN32__
int gettimeofday (struct timeval *tp, void *tz)
{
	struct _timeb timebuffer;
	_ftime (&timebuffer);
	tp->tv_sec = timebuffer.time;
	tp->tv_usec = timebuffer.millitm * 1000;
	return 0;
}
#endif

void
usage(char *s) {
  fprintf(stderr, "Usage: %s <file.cfg> -dag <dag_file>\n", s);
  fprintf(
    stderr,
    "       %s <file.cfg> -wf <wf_file> <data_file> [transcript_file] [-name WFNAME]\n",
    s);
  exit(1);
}
int
checkUsage(int argc, char **argv) {
  if ((argc < 4) || (argc > 8)) {
    usage(argv[0]);
  }
  return 0;
}

int
main(int argc, char *argv[]) {
  diet_wf_desc_t *profile;
  char *wfFileName, *dataFileName, *transcriptFileName, *wfName, *dagFileName;
  wf_level_t wfType;
  char wfTypeName[10];
  struct timeval t1, t2;

  checkUsage(argc, argv);

  if (diet_initialize(argv[1], argc, argv)) {
    fprintf(stderr, "DIET initialization failed !\n");
    return 1;
  }

  wfFileName = (char *) NULL;
  dataFileName = (char *) NULL;
  transcriptFileName = (char *) NULL;
  wfName = (char *) NULL;
  dagFileName = (char *) NULL;

  if (!strcmp(argv[2], "-dag")) {
    strcpy(wfTypeName, "dag");
    wfType = DIET_WF_DAG;
    if (argc > 4) {
      usage(argv[0]);
    }
  } else if (!strcmp(argv[2], "-wf")) {
    strcpy(wfTypeName, "workflow");
    wfType = DIET_WF_FUNCTIONAL;
    if (argc < 5) {
      usage(argv[0]);
    }
  } else {
    usage(argv[0]);
  }

  if (wfType == DIET_WF_FUNCTIONAL) {
    int curPos = 5;
    wfFileName = argv[3];
    dataFileName = argv[4];
    if ((argc > curPos) && (strcmp(argv[curPos], "-name"))) {
      transcriptFileName = argv[curPos];
      curPos++;
    }
    if ((argc > curPos) && (!strcmp(argv[curPos], "-name"))) {
      curPos++;
      if (argc > curPos) {
        wfName = argv[curPos];
      } else {
        wfName = "";
      }
    } else {
#ifdef __WIN32__
        _splitpath(wfFileName,NULL,NULL,wfName,NULL);
#else
        wfName = basename(wfFileName);
#endif

    }
  } else {      /* DIET_WF_DAG */
    dagFileName = argv[3];
  }

  gettimeofday(&t1, NULL);


  if (wfType == DIET_WF_FUNCTIONAL) {
    /*
     * Allocate the workflow profile
     */
    profile = diet_wf_profile_alloc(wfFileName, wfName, wfType);
    /*
     * For functional workflows ONLY
     * Defines which file is used to provide the data to instanciate the wf
     */
    diet_wf_set_data_file(profile, dataFileName);

    /*
     * For workflow restart
     * Defines which file is used to store the execution transcriptFileName
     * (file will be overwritten if existing)
     */
    diet_wf_set_transcript_file(profile, transcriptFileName);
  } else {      /* DIET_WF_DAG */
    /*
     * Allocate the dag profile
     */
    profile = diet_wf_profile_alloc(dagFileName, "test", wfType);
  }

  printf("Try to execute the %s\n", wfTypeName);
  if (!diet_wf_call(profile)) {
    float time;
    gettimeofday(&t2, NULL);
    time =
      (t2.tv_sec - t1.tv_sec) + ((float) (t2.tv_usec - t1.tv_usec)) / 1000000;
    printf("The %s submission succeed / time= %f s\n", wfTypeName, time);

    if (wfType == DIET_WF_FUNCTIONAL) {
      printf("Save data in data_out.xml\n");
      if (diet_wf_save_data_file(profile, "data_out.xml")) {
        printf("Could not save data file\n");
      }
    }

    printf("Display results:\n");
    if (diet_wf_print_results(profile) != 0) {
      printf("Could not display results\n");
    }
  } else {
    printf("The %s submission failed\n", wfTypeName);
  }

  if (transcriptFileName == NULL) {
    diet_wf_free(profile);
  } else {
    printf("Save transcript of workflow execution\n");
    diet_wf_save_transcript_file(profile, transcriptFileName);
  }
  diet_finalize();
  fflush(stdout);
  return 0;
} /* main */
