/**
 * @file Container.cc
 *
 * @brief  Container implementation
 *
 * @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "DIET_data_internal.hh"
#include "marshalling.hh"
#include "debug.hh"
#include "Container.hh"
#include "DagdaFactory.hh"
#include "DagdaImpl.hh"
#include "ORBMgr.hh"

using namespace std;

Container::Container(const char *dataID, Dagda_ptr dataMgr,
                     DataRelationMgr *relMgr)
  : myID(dataID), notFound(false), nbOfElements(0) {
  myMgr = dataMgr;
  myRelMgr = relMgr;

  // check if container is already existing on the local data mgr
  // EXISTING means registered but not all elements of the container are
  // necessarily present locally
  if (!myMgr->lclIsDataPresent(dataID)) {
    notFound = true;
  }
}

Container::~Container() {
}

void
Container::addData(const char *dataID, long index, long flag) {
  // TRACE_TEXT(TRACE_ALL_STEPS, "Container: Add the data " << dataID
  // << "(flag=" << flag << ") to container " << myID << endl);
  if (notFound) {
    throw Dagda::DataNotFound(myID.c_str());
  }
  myRelMgr->addRelation(myID, dataID, index, flag);
}

void
Container::remData(long index) {
  if (notFound) {
    throw Dagda::DataNotFound(myID.c_str());
  }
  myRelMgr->remRelation(myID, index);
}

int
Container::size() {
  if (notFound) {
    throw Dagda::DataNotFound(myID.c_str());
  }
  return (myRelMgr->getRelationMaxIndex(myID) + 1);
}

void
Container::getAllElements(SeqString &dataIDSeq, SeqLong &flagSeq,
                          bool ordered) {
  if (notFound) {
    throw Dagda::DataNotFound(myID.c_str());
  }
  int _size = size();
  try {
    dataIDSeq.length(_size);
    flagSeq.length(_size);
  } catch (...) {
    cerr << "Container::getAllElements exception in initialization" << endl;
    throw;
  }
  try {
    myRelMgr->getRelationList(myID, dataIDSeq, flagSeq, ordered);
  } catch (...) {
    cerr << "Container::getAllElements exception in getRelationList" << endl;
    throw;
  }
} // getAllElements

/**
 * Send a container
 * Called from a dagda agent that contains the container data but that does not
 * necessarily contains all the elements. So the source for the elements is got
 * from a call to the platform.
 */
char *
Container::send(const char *destName, bool sendData) {
  Dagda_ptr dest = ORBMgr::getMgr()->resolve<Dagda, Dagda_ptr>(DAGDACTXT,
                                                               destName);

  SeqString *dataIDSeq = new SeqString();
  SeqLong *flagSeq = new SeqLong();
  this->getAllElements(*dataIDSeq, *flagSeq, true);

  TRACE_TEXT(TRACE_ALL_STEPS, "Sending container " << myID << " ("
                                                   << dataIDSeq->length() <<
             " elements / sendData="
                                                   << sendData << ")" << endl);

  for (unsigned int ix = 0; ix < dataIDSeq->length(); ++ix) {
    if ((*dataIDSeq)[ix] != NULL) {
      const char *eltID = (*dataIDSeq)[ix];
      if (sendData) {
        corba_data_desc_t *eltDesc;
        try {
          eltDesc = myMgr->pfmGetDataDesc(eltID);
        } catch (Dagda::DataNotFound &ex) {
          WARNING(
            "Missing element " << eltID << " in container " << myID << endl);
          continue;  // skip this element
        }
        corba_data_t eltData;
        eltData.desc = *eltDesc;
        // Dagda_var srcMgr = Dagda::_narrow(ORBMgr::stringToObject(eltDesc->dataManager));
        Dagda_ptr srcMgr =
          ORBMgr::getMgr()->resolve<Dagda,
                                    Dagda_ptr>(string(eltDesc->dataManager));
        string srcName = srcMgr->getID();
        dest->lclAddData(srcName.c_str(), eltData);
      }
      // add relationship container-data to destination mgr
      dest->lclAddContainerElt(myID.c_str(), eltID, ix, (*flagSeq)[ix]);
    } else {  // dataIDSeq[ix] = NULL
      dest->lclAddContainerElt(myID.c_str(), "", ix, (CORBA::Long) 1);
    }
  }
  return CORBA::string_dup(myID.c_str());
} // send
