/**
* @file  WfDataWriter.cc
* 
* @brief  Abstract interface and implementations of data writers 
* 
* @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr)
* 
* @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/


#include "WfDataWriter.hh"
#include <iostream>
#include "WfPortAdapter.hh"  // for voidRef

/*****************************************************************************/
/*                          WfDataWriter class                               */
/*****************************************************************************/

WfDataWriter::WfDataWriter(std::ostream& output)
  : myOutput(output) {
}

WfDataWriter::~WfDataWriter() {
}

void
WfDataWriter::rawValue(const std::string& valueStr) {
  myOutput << valueStr;
}

void
WfDataWriter::rawValue(void * valuePtr,
                       WfCst::WfDataType valueType) {
  switch (valueType) {
  case WfCst::TYPE_DOUBLE:
    myOutput << *((double *) valuePtr);
    break;
  case WfCst::TYPE_INT:
    myOutput << *((int *) valuePtr);
    break;
  case WfCst::TYPE_LONGINT:
    myOutput << *((long *) valuePtr);
    break;
  case WfCst::TYPE_FLOAT:
    myOutput << *((float *) valuePtr);
    break;
  case WfCst::TYPE_SHORT:
    myOutput << *((short *) valuePtr);
    break;
  case WfCst::TYPE_CHAR:
  case WfCst::TYPE_PARAMSTRING:
  case WfCst::TYPE_STRING:
  case WfCst::TYPE_FILE:
    myOutput << (char *) valuePtr;
    break;
  case WfCst::TYPE_MATRIX:
  case WfCst::TYPE_CONTAINER:
  case WfCst::TYPE_UNKNOWN:
    myOutput << "#ERROR DATA RAW VALUE WRITER#";
  }
}

/*****************************************************************************/
/*                        WfXMLDataWriter class                              */
/*****************************************************************************/

WfXMLDataWriter::WfXMLDataWriter(std::ostream& output)
  : WfDataWriter(output) {}

void
WfXMLDataWriter::startContainer() {
  myOutput << "<array>\n";
}

void
WfXMLDataWriter::startContainer(const std::string& dataID) {
  myOutput << "<array dataId=\"" << dataID << "\">\n";
}

void
WfXMLDataWriter::endContainer() {
  myOutput << "</array>\n";
}

void
WfXMLDataWriter::itemValue(const std::string& valueStr) {
  // FIXME tag should not be added if no array was open previously
  myOutput << "<item>";
  rawValue(valueStr);
  myOutput << "</item>\n";
}

void
WfXMLDataWriter::itemValue(const std::string& valueStr,
                           const std::string& dataID) {
  myOutput << "<item dataId=\"" << dataID << "\">";
  rawValue(valueStr);
  myOutput << "</item>\n";
}

void
WfXMLDataWriter::itemValue(void * valuePtr, WfCst::WfDataType valueType) {
  myOutput << "<item>";
  rawValue(valuePtr, valueType);
  myOutput << "</item>\n";
}

void
WfXMLDataWriter::itemValue(void * valuePtr, WfCst::WfDataType valueType,
                           const std::string& dataID) {
  myOutput << "<item dataId=\"" << dataID << "\">";
  rawValue(valuePtr, valueType);
  myOutput << "</item>\n";
}

void
WfXMLDataWriter::matrix(void * valuePtr, WfCst::WfDataType valueType,
                        unsigned int nbRows, unsigned int nbColumns,
                        diet_matrix_order_t order) {
  // TODO
}

void
WfXMLDataWriter::voidElement() {
}

void
WfXMLDataWriter::error() {
}

/*****************************************************************************/
/*                        WfListDataWriter class                             */
/*****************************************************************************/


WfListDataWriter::WfListDataWriter(std::ostream& output)
  : WfDataWriter(output), firstElt(true) {
}

void
WfListDataWriter::startContainer() {
  if (!firstElt) {
    myOutput << ", ";
  }
  myOutput << "(";
  firstElt = true;
}

void
WfListDataWriter::endContainer() {
  myOutput << ")";
  firstElt = false;
}

void
WfListDataWriter::itemValue(const std::string& valueStr) {
  if (!firstElt) {
    myOutput << ", ";
  }
  rawValue(valueStr);
  firstElt = false;
}

void
WfListDataWriter::itemValue(void * valuePtr, WfCst::WfDataType valueType) {
  rawValue(valuePtr, valueType);
}

void
WfListDataWriter::matrix(void * valuePtr, WfCst::WfDataType valueType,
                         unsigned int nbRows, unsigned int nbColumns,
                         diet_matrix_order_t order) {
  // TODO
}

void
WfListDataWriter::voidElement() {
  myOutput << WfVoidAdapter::voidRef;
}

void
WfListDataWriter::error() {
}
