/**
* @file EnvParser.cc
* 
* @brief  Environment variables parser 
* 
* @author  Haikel Guemar (haikel.guemar@sysfera.com)
* 
* @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/



#include "EnvParser.hh"
#include "CommonParser.hh"

#include <algorithm>
#include <functional>


// utils

/**
 * @author hguemar <hguemar@sysfera.com>
 *
 * @class CheckEnvEntry
 * @brief functor testing if two environment entries conflicts
 * @internal
 */
class CheckEnvEntry : public std::unary_function<std::string, bool>
{
private:
  const std::pair<std::string, std::string>& s1;
public:
  CheckEnvEntry(const std::pair<std::string, std::string>& entry)
    : s1(entry) {}

  bool operator() (const std::pair<std::string, std::string>& s2) const
  {
    return ((s1.first == s2.first) || (s1.second == s2.second))
      ? true : false;
  }
};


// public methods

EnvParser::EnvParser(char *envp[], ConfigMap& config) :
  config_(config)
{
  parseEnv(envp);
}

void EnvParser::parse()
{
  Splitter split('=');
  std::string key, value;

  for (ConfigMap::iterator it1 = config_.begin();
       it1 != config_.end(); ++it1) {
    Env::iterator it2 =
      std::find_if (env_.begin(), env_.end(), StartsWith(it1->second));
    if (env_.end() != it2) {
      split.reset(*it2);

      std::string& token = split();
      key.swap(token);
      token = split();
      value.swap(token);

      debug << "Environment variable " << key
            << " : " << value << "\n";

      options_[key] = value;

      env_.erase(it2);
    }
  }
}

// use of operator[] is more idiomatic
std::string& EnvParser::operator[](const std::string& key)
{
  return options_[key];
}


const std::string&
EnvParser::operator[](const std::string& key) const
{
  return options_[key];
}


const std::map<std::string, std::string>&
EnvParser::getConfiguration() const
{
  return options_;
}

// private methods
void EnvParser::lint() const
{
  for (ConfigMap::const_iterator it1 = config_.begin();
       it1 != config_.end();) {
    ConfigMap::const_iterator it = it1++;
    ConfigMap::const_iterator it2 =
      std::find_if (it1, config_.end(), CheckEnvEntry(*it));

    if (config_.end() != it2) {
      throw EnvConfigError("");
    }
  }
}


void EnvParser::parseEnv(char *envp[])
{
  int i = 0;
  while (0 != envp[i]) {
    env_.push_back(envp[i]);
    i++;
  }
}


