/**
 * @file ec2wrapper.h
 *
 * @brief  Batch System implementation for Amazon EC2 compatible clouds
 *
 * @author Adrian Muresan (adrian.muresan@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */
/****************************************************************************/
/* Batch System implementation for Amazon EC2 compatible clouds             */
/* Contains a set of helper functions for communicating with Eucalyptus     */
/* (and Amazon EC2)                                                         */
/****************************************************************************/

#ifndef _EC2WRAPPER_H_
#define _EC2WRAPPER_H_

#include "soapH.h"
#include "plugin/smdevp.h"
#include "plugin/wsseapi.h"
#include "stdsoap2.h"



#ifdef __cplusplus
extern "C" {
#endif



// Allocates and initializes a new soap structure with default settings for Eucalyptus i.e. with the SOAP_XML_CANONICAL | SOAP_XML_INDENT flags
extern struct soap *
soap_default_new();

// Reads a private key located at the specified path and returns it in the pointer parameter.
// Returns 0 - OK; !=0 - error.
extern int
PEM_read_PK(char *pem_path, EVP_PKEY **pk_out);

// Reads an X509 certificate located at the specified path and returns it in the pointer parameter.
// Returns 0 - OK; !=0 - error
extern int  
PEM_read_X509_cert(char *pem_path, X509 **cert_out);

// Adds a security header to the soap message containing:
// 1. BinarySecurityToken — Contains the X.509 certificate in base64 encoded PEM format
// 2. Signature — Contains an XML digital signature created using the canonicalization, signature algorithm, and digest method (RSA-SHA1)
// 3. Timestamp — Requests to Amazon EC2 are valid within 5 minutes of this value to help prevent replay attacks
// Parameters:
// 1. the soap message
// 2. the private key
// 3. the X509 certificate
// Returns 0 - OK; !=0 - error
extern int
soap_wsse_add_security_header(struct soap *soap, EVP_PKEY *pk, X509 *cert);

#ifdef __cplusplus
}
#endif

#endif // _EC2WRAPPER_H_

/* end of ec2wrapper.h */
