/**
* @file  MetaDagr.hh
* 
* @brief  A structure containing multiple dags having inter-relationships
* 
* @author  Benjamin ISNARD (Benjamin.Isnard@ens-lyon.fr)
* 
* @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/


#ifndef _METADAG_HH_
#define _METADAG_HH_

#include <map>
#include <string>
#include "NodeSet.hh"
#ifdef WIN32
#define DIET_API_LIB __declspec(dllexport)
#else
#define DIET_API_LIB
#endif
class Dag;
class DagScheduler;

/*****************************************************************************/
/*                            METADAG CLASS                                  */
/*****************************************************************************/

/**
 * MetaDag class
 * Used to manage inter-dag relationships
 */

class DIET_API_LIB MetaDag : public NodeSet {
public:
  explicit MetaDag(const std::string& id);

  virtual ~MetaDag();

  /**
   * Get the Id
   */
  const std::string&
  getId();

  /**
   * Add a dag to the metaDag
   * (the dag is supposed not yet executed)
   * @param dag the dag ref
   */
  void
  addDag(Dag * dag);

  /**
   * Get a dag by ID
   * @param dagId the dag identifier (std::string)
   */
  Dag *
  getDag(const std::string& dagId) throw(WfStructException);

  /**
   * Remove a dag from the metaDag
   * @param dagId the dag identifier (std::string)
   */
  void
  removeDag(const std::string& dagId) throw(WfStructException);


  /**
   * Remove a dag from the metaDag but just ignore missing element requested
   * (only used for cleaning after a DAG execution to avoid leftover
   * intermediate dags generated by loops and such)
   *
   * @param dagId the dag identifier (std::string)
   *
   */
  void removeDagIgnoreMissing(const std::string& dagId);

  /**
   * Nb of dags
   */
  int
  getDagNb();

  /**
   * Set the release flag and returns the metadag completion status
   * TRUE => destroy metadag when last dag is done
   * FALSE => do not destroy metadag when last dag is done
   */
  void
  setReleaseFlag(bool release);
  void
  setReleaseFlag(bool release, bool& isDone);

  /**
   * Set the current dag used when parsing node references
   * (node references without a dag prefix are found using this default dag)
   * @param dag the default dag ref
   */
  void
  setCurrentDag(Dag * dag);

  /**
   * Search a node reference among the nodes of the metadag's dags
   * @param nodeId  the node reference ('nodeId' or 'dagId:nodeId')
   * @return pointer to node (does not return NULL)
   */
  virtual WfNode*
  getNode(const std::string& nodeId) throw(WfStructException);

  /**
   * Manages dag termination
   * (when dag terminates the scheduler calls this handler and does NOT delete dag)
   * @param dag the ref of the terminated dag
   */
  virtual void
  handlerDagDone(Dag * dag);

  /**
   * Check if metadag is completed
   * (for destruction)
   */
  bool
  isDone();

  /**
   * Cancel all dags
   */
  void
  cancelAllDags(DagScheduler * scheduler = NULL);

protected:

  /**
   * Metadag ID
   */
  std::string myId;

  /**
   * Map containing all the dag refs
   */
  std::map<std::string, Dag*> myDags;

  /**
   * Pointer to the current dag (default dag used when parsing refs to nodes)
   */
  Dag * currDag;

  /**
   * Counter of not finished dags
   */
  int dagTodoCount;

  /**
   * Release flag
   */
  bool releaseFlag;

  /**
   * Cancelled flag
   */
  bool cancelFlag;

  /**
   * Critical section
   */
  omni_mutex myLock;

  void
  lock();

  void
  unlock();

  /**
   * Not applicable to this class
   */
  virtual void
  checkPrec(NodeSet* contextNodeSet) throw(WfStructException);
};

#endif   /* not defined _METADAG_HH. */
