/**
 * @file  DIET_mutex.cc
 *
 * @brief  DIET mutex interface for multi-threaded server applications (source code)
 *
 * @author  Philippe COMBES (Philippe.Combes@ens-lyon.fr)
 *          Bert VAN HEUKELOM (Bert.Van-Heukelom@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "DIET_mutex.h"

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include "debug.hh"
#include "omnithread.h"


static omni_mutex **MUTEX_FIELD = NULL;
static int MUTEXCOUNT = 0;
static int INITIALIZED = 0;


#define MUTEX_ERROR(formatted_text)                     \
  ERROR_DEBUG(__FUNCTION__ << ": " << formatted_text, )

#define MUTEX_CHECK_INIT()                                      \
  if (!INITIALIZED) {                                           \
    MUTEX_ERROR("diet_mutex_initialize has not been called");   \
  }

#define MUTEX_CHECK_VALIDITY(i)                 \
  if ((i) >= MUTEXCOUNT) {                      \
    MUTEX_ERROR("invalid mutex");               \
  }



void
diet_mutex_initialize() {
  int i;
  MUTEX_FIELD = (omni_mutex **) malloc(10 * sizeof(omni_mutex));
  for (i = 0; i < 10; i++) {
    MUTEX_FIELD[i] = NULL;
  }
  MUTEXCOUNT = 10;
  INITIALIZED = 1;
}

void
diet_mutex_create(int *ret) {
  int i;
  omni_mutex **temp;

  MUTEX_CHECK_INIT();

  for (i = 0; i < MUTEXCOUNT; i++) {
    if (MUTEX_FIELD[i] == NULL) {
      MUTEX_FIELD[i] = new omni_mutex();
      *ret = i;
      return;
    }
  }

  temp = (omni_mutex **) malloc((10 + MUTEXCOUNT) * sizeof(omni_mutex));
  for (i = 0; i < MUTEXCOUNT; i++) {
    temp[i] = MUTEX_FIELD[i];
  }
  for (i = 0; i < 10; i++) {
    temp[i + MUTEXCOUNT] = NULL;
  }
  free(MUTEX_FIELD);
  MUTEX_FIELD = temp;

  MUTEX_FIELD[MUTEXCOUNT] = new omni_mutex();
  *ret = MUTEXCOUNT;

  MUTEXCOUNT += 10;
} // diet_mutex_create

void
diet_mutex_free(int *i) {
  MUTEX_CHECK_INIT();
  MUTEX_CHECK_VALIDITY(*i);

  delete (MUTEX_FIELD[*i]);

  MUTEX_FIELD[*i] = NULL;
  *i = 0;
}

void
diet_mutex_lock(int i) {
  MUTEX_CHECK_INIT();
  MUTEX_CHECK_VALIDITY(i);

  MUTEX_FIELD[i]->lock();
}

void
diet_mutex_unlock(int i) {
  MUTEX_CHECK_INIT();
  MUTEX_CHECK_VALIDITY(i);

  MUTEX_FIELD[i]->unlock();
}

void
diet_mutex_finalize() {
  MUTEX_CHECK_INIT();

  free(MUTEX_FIELD);
  INITIALIZED = 0;
}

void
diet_thread_sleep(int m, int n) {
  omni_thread *myThread = NULL;
  myThread = omni_thread::self();
  myThread->sleep(m, n);
}

void
diet_thread_yield() {
  omni_thread *myThread = NULL;
  myThread = omni_thread::self();
  myThread->yield();
}

int
diet_thread_id() {
  omni_thread *myThread = NULL;
  myThread = omni_thread::self();
  return myThread->id();
}
