/**
 * @file  DataRelationMgr.hh
 *
 * @brief  Class used to manage relationships between data items (used for containers in DAGDA)
 *
 * @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef __DATARELATIONMGR_HH__
#define __DATARELATIONMGR_HH__

#include "DIET_data.h"
#include "common_types.hh"

#include <map>
#include <string>

/**
 * Class DataRelationMgr
 *
 * This class is used to store and manage relationships between data items of
 * a DAGDA agent. A relationship is simply a link between two data items with
 * two additionnal fields 'index' and 'flag'. 'index' is used to keep an order
 * in the links from a given item, and 'flag' is used to store specific info
 * about the linked item (e.g. size of the item).
 */

class DataRelationMgr {
public:
  DataRelationMgr();

  /**
   * Add a relationship between two data items
   * @param dataID1 id of first item
   * @param dataID2 id of second item
   * @param index   index of the link
   * @param flag    specific info about linked item
   */
  void
  addRelation(const std::string &dataID1, const std::string &dataID2,
              long index, long flag = 0);

  /**
   * Remove the relationship given parent data id and index
   * @param dataID1 id of first item
   * @param index   index of the link
   */
  void
  remRelation(const std::string &dataID1, long index);

  /**
   * Remove all relationships for a given item
   * @param dataID  id of the item
   * @param reverse if true, will consider links in BOTH WAYS
   */
  void
  remAllRelation(const std::string &dataID, bool reverse = false);

  /**
   * Get the total nb of relations for a given dataID
   * @param dataID  id of the container
   */
  unsigned int
  getRelationNb(const std::string &dataID);

  /**
   * Get the maximum index value for a given dataID (indexes start from 0)
   * @param dataID  id of the container
   */
  unsigned int
  getRelationMaxIndex(const std::string &dataID);

  /**
   * Get the relationships from a given item (using index order)
   *
   * The two sequences returned must be pre-allocated with enough space
   * to store all the relationships found
   *
   * The sequence may contain NULL elements if some indexes are missing
   * in the database.
   *
   * @param dataID      id of the item
   * @param dataIDList  a sequence of std::string containing the IDs
   * @param flagList    a sequence of integers containing the flags
   * @param ordered     if false, will not use the index field to sort
   */
  void
  getRelationList(const std::string &dataID, SeqString &dataIDList,
                  SeqLong &flagList, bool ordered = true);

  /**
   * Display all relationships (for debug)
   */
  void
  displayContent();

protected:
  typedef struct {
    std::string ID;
    long index;
    long flag;
  } dataRelationValue_t;

  std::multimap<std::string, dataRelationValue_t> myMap;

  /**
   * Critical section
   */
  omni_mutex myLock;
};  // end class DataRelationMgr

#endif /* ifndef __DATARELATIONMGR_HH__ */
