/**
 * @file EventTypes.hh
 *
 * @brief   Template event classes (inherit from EventBase)
 *
 * @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef _EVENTTYPES_HH_
#define _EVENTTYPES_HH_

#include <iostream>
#include <sstream>
#include <string>
#include "EventBase.hh"
#include "EventManager.hh"

namespace events {
/**
 * @class EventFrom
 * Used for all events generated by one object of the model about its own state
 * @param SOURCE_TYPE   the class of the model object
 *                      --> must implement string toString() const
 */

template <class SOURCE_TYPE, class MSG_CLASS>
class EventFrom : public EventBase {
public:
  EventFrom(EventBase::Severity severity, std::string message,
            std::string data, const SOURCE_TYPE * src)
  : EventBase(severity), myMessage(message), myData(data), mySource(src) {
  }

  std::string
  getMessage() const {
    return myMessage;
  }

  std::string
  getData() const {
    return myData;
  }

  const SOURCE_TYPE *
  getSource() const {
    return mySource;
  }

  virtual std::string
  toString() const;

private:
  std::string myMessage;
  std::string myData;
  const SOURCE_TYPE *mySource;
  static const int myMsgCode;
};

template <class SOURCE_TYPE, class MSG_CLASS>
std::string
EventFrom<SOURCE_TYPE, MSG_CLASS>::toString() const {
  std::ostringstream str;
  str << EventBase::toString() << " : " << myMessage << " : ";
  str << "FROM " << mySource->toString() << " : " << myData << "/END";
  return str.str();
}

/**
 * @class EventStandardMsg
 * @param SOURCE_TYPE   the type of object producing the event
 * @param MSG_TYPE      integer code for the message type (declared in source obj class)
 */
template <class SOURCE_TYPE, int MSG_TYPE>
class EventStandardMsg {
private:
  static const int myMsgCode;
};

template <class SOURCE_TYPE, int MSG_TYPE>
const int
EventStandardMsg<SOURCE_TYPE, MSG_TYPE>::myMsgCode = MSG_TYPE;


template<class SOURCE_TYPE, int MSG_TYPE>
static void
sendEventFrom(const SOURCE_TYPE *src, std::string msg,
              std::string data, EventBase::Severity severity) {

	EventFrom<SOURCE_TYPE, EventStandardMsg<SOURCE_TYPE, MSG_TYPE> > *event =
			new EventFrom<SOURCE_TYPE, EventStandardMsg<SOURCE_TYPE, MSG_TYPE> >(severity, msg, data, src);
	EventManager::getEventMgr()->sendEvent(event);
	delete event;
}

/**
 * @class EventCreateObject
 * @param OBJ_TYPE      the type of object created
 * @param PARENT_TYPE   the type of the parent of the object created
 */

template <class OBJ_TYPE, class PARENT_TYPE>
class EventCreateObject : public EventBase {
public:
  EventCreateObject(OBJ_TYPE * obj, PARENT_TYPE * parent)
  : myObj(obj), myParent(parent) {
  }

  const OBJ_TYPE *
  getObject() const {
    return myObj;
  }

  const PARENT_TYPE *
  getParent() const {
    return myParent;
  }

  virtual std::string
  toString() const;

private:
  OBJ_TYPE *myObj;
  PARENT_TYPE *myParent;
};

template <class OBJ_TYPE, class PARENT_TYPE>
std::string
EventCreateObject<OBJ_TYPE, PARENT_TYPE>::toString() const {
  if (myParent == NULL) {
    return "CREATION OF " + myObj->toString();
  } else {
    return "CREATION OF " + myObj->toString() +
           " [PARENT=" + myParent->toString() + "]";
  }
}


/**
 * @class EventCreateDirectedLink
 * @param SRC_TYPE      the type of the source object
 * @param DEST_TYPE     the type of the destination object
 */
template <class SRC_TYPE, class DEST_TYPE>
class EventCreateDirectedLink : public EventBase {
public:
  EventCreateDirectedLink(SRC_TYPE * src, DEST_TYPE * dest)
  : mySrc(src), myDest(dest) {
  }

  const SRC_TYPE *
  getSource() const {
    return mySrc;
  }

  const DEST_TYPE *
  getDestination() const {
    return myDest;
  }

  virtual std::string
  toString() const;

private:
  SRC_TYPE *mySrc;
  DEST_TYPE *myDest;
};

template <class SRC_TYPE, class DEST_TYPE>
std::string
EventCreateDirectedLink<SRC_TYPE, DEST_TYPE>::toString() const {
  return "CREATION OF LINK FROM=" + mySrc->toString() +
         " TO=" + myDest->toString();
}
}
#endif  // _WFEVENT_HH_
