/**
 * @file HEFTScheduler.hh
 *
 * @brief  The MA DAG HEFT scheduler
 *
 * @author  Abdelkader AMAR (Abdelkader.Amar@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef _HEFTSCHEDULER_HH_
#define _HEFTSCHEDULER_HH_

#include <map>
#include <string>
#include <vector>
#include <set>
#include "Dag.hh"
#include "WfScheduler.hh"

namespace madag {
class HEFTScheduler : public WfScheduler {
public:
  HEFTScheduler();

  virtual ~HEFTScheduler();

  /**
   * Set the priority of nodes using b-level algorithm
   */
  virtual void
  setNodesPriority(const wf_response_t *wf_response, Dag *dag);

  /**
   * Set the EFT of nodes using HEFT algorithm
   *
   * @param orderedNodes  vector of nodes ordered by decreasing priority
   * @param wf_response   the estimates given by the MA
   * @param dag           the dag
   * @param initTime      dag starting time in ms (relative)
   */
  virtual void
  setNodesEFT(std::vector<DagNode *> &orderedNodes,
              const wf_response_t *wf_response,
              Dag *dag,
              double initTime);

private:

  /**
   * Get the value of estimated job duration from wf response
   * for a given service and a given server
   * @param wf_response the wf response structure
   * @param pbIndex the index of the problem (service)
   * @param srvIndex the index of the server
   */
  double
  getNodeDurationEst(const wf_response_t *wf_response,
                     unsigned int pbIndex,
                     unsigned int srvIndex);

  /**
   * Computes the average value of node workload across the Seds
   */
  void
  computeNodeWeights(const wf_response_t *wf_response, Dag *dag);

  /**
   * rank the node upward
   * @param n the top node to rank.
   */
  void
  rank(DagNode *n, std::set<std::string> *alreadyRanked);

  /**
   * Stores all the nodes earliest finish times
   */
  std::map<std::string, double> AFT;
};
}

#endif   /* not defined _HEFTSCHEDULER_HH */
