/**
 * @file BindService.cc
 *
 * @brief  A thread which updates the MultiMA links
 *
 * @author  Sylvain DAHAN (Sylvain.Dahan@lifc.univ-fcomte.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "MasterAgentImpl.hh"
#include "BindService.hh"

#if HAVE_MULTI_MA

#include "ms_function.hh"
#include "debug.hh"
#include "ORBMgr.hh"
#include <cstdio>
#include <sys/types.h>
#ifndef __WIN32__
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#else
#include <WinSock.h>
#include <Winsock2.h>
#include <windows.h>
#endif
#include <cstring>
#include <cerrno>

extern unsigned int TRACE_LEVEL;

void
BindService::run(void *ptr) {
  bool doNotStop = true;
  char *ior = this->ior;
  this->ior = NULL;
  while (doNotStop) {
    struct sockaddr_in clientAddr;
    socklen_t clientLen = static_cast<socklen_t>(sizeof(clientAddr));
    int newSockFd = accept(listenSocket, (struct sockaddr *) &clientAddr,
                           &clientLen);
    if (newSockFd < 0) {
      if (errno == EBADF) {
        doNotStop = false;
        TRACE_TEXT(TRACE_ALL_STEPS, "bind service closed");
        continue;
      } else {
        WARNING("error on accept (bind service): " << strerror(errno));
        continue;
      }
    }
    int n = write(newSockFd, ior, strlen(ior));
    if (n < 0) {
      WARNING("error when writing to socket (bind service): "
              << strerror(errno));
    }
    n = close(newSockFd);
    if (n < 0) {
      WARNING("error when writing to socket (bind service): "
              << strerror(errno));
    }
  }
  free(ior);
} // run

BindService::~BindService() {
}

BindService::BindService(MasterAgentImpl *ma, unsigned int port) {
  ior = CORBA::string_dup(ORBMgr::getMgr()->getIOR(ma->_this()).c_str());

  struct sockaddr_in serverAddr;
  listenSocket = socket(AF_INET, SOCK_STREAM, 0);
  if (listenSocket < 0) {
    ERROR_DEBUG("opening bind service socket: " << strerror(errno) << "\n",;
          );
  }
  memset((char *) &serverAddr, 0, sizeof(serverAddr));
  serverAddr.sin_family = AF_INET;
  serverAddr.sin_addr.s_addr = INADDR_ANY;
  serverAddr.sin_port = htons(port);
  if (bind(listenSocket, (struct sockaddr *) &serverAddr,
           sizeof(serverAddr)) < 0) {
    ERROR_DEBUG("in binding the bind service socket: " << strerror(errno) << "\n",;
          );
  }
  listen(listenSocket, 5);

  TRACE_TEXT(TRACE_ALL_STEPS, "bind service open\n");
  start();
}


MasterAgent_ptr
BindService::lookup(const char *addr) {
  assert(addr != NULL);
  char hostname[256];
  strncpy(hostname, addr, sizeof(hostname) - 1);
  hostname[sizeof(hostname) - 1] = '\0';
  char *idx = strchr(hostname, ':');
  int portNo = 0;
  if (idx != NULL) {
    idx[0] = '\0';
    portNo = atoi(idx + 1);
  }
  if (portNo == 0) {
    TRACE_TEXT(TRACE_ALL_STEPS, addr << " is not a valid address\n");
    return MasterAgent::_nil();
  }

  int sockfd = socket(AF_INET, SOCK_STREAM, 0);
  if (sockfd < 0) {
    TRACE_TEXT(TRACE_ALL_STEPS, " opening socket:" << strerror(errno) << "\n");
    return MasterAgent::_nil();
  }

  struct hostent *server = gethostbyname(hostname);
  if (server == NULL) {
    TRACE_TEXT(TRACE_ALL_STEPS, " no such host: " << hostname << "\n");
    return MasterAgent::_nil();
  }

  struct sockaddr_in servAddr;
  memset((char *) &servAddr, 0, sizeof(servAddr));
  servAddr.sin_family = AF_INET;
  memmove((char *) &servAddr.sin_addr.s_addr,
          (char *) server->h_addr,
          server->h_length);  // use memmove instead of bcopy
  servAddr.sin_port = htons(portNo);

  if (connect(sockfd, (const sockaddr *) &servAddr, sizeof(servAddr)) < 0) {
    TRACE_TEXT(TRACE_ALL_STEPS, " not connecting:" << strerror(errno) << "\n");
    return MasterAgent::_nil();
  }
  char buffer[2048];
  memset(buffer, 0, sizeof(buffer));  // use memset instead of bzero
  int n = read(sockfd, buffer, (sizeof(buffer) - 1));
  if (n < 0) {
    TRACE_TEXT(TRACE_ALL_STEPS,
               " reading from socket:" << strerror(errno) << "\n");
    close(sockfd);
    return MasterAgent::_nil();
  }
  close(sockfd);
  CORBA::Object_var obj = ORBMgr::getMgr()->resolveObject(buffer);
  if (CORBA::is_nil(obj)) {
    TRACE_TEXT(TRACE_ALL_STEPS, "is nil: " << buffer << "\n");
    return MasterAgent::_nil();
  }
  return MasterAgent::_narrow(obj);
} // lookup

#endif  // HAVE_MULTI_MA
