/**
* @file CmdException.hh
* 
* @brief  Command line exceptions 
* 
* @author  Haikel Guemar (haikel.guemar@sysfera.com)
* 
* @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/

#ifndef _CMD_EXCEPTION_HH_
#define _CMD_EXCEPTION_HH_

#include <exception>
#include <sstream>
#include <string>

/**
 * @author hguemar <hguemar@sysfera.com>
 *
 * @class CmdParserError
 * @brief Base class for all CmdParser exceptions
 */
class CmdParserError : public std::exception {
};

/**
 * @author hguemar <hguemar@sysfera.com>
 *
 * @class CmdConfigError
 * @brief CmdParser ill-formed configuration
 */
class CmdConfigError : public CmdParserError {
public:
  explicit CmdConfigError(const std::string& msg)
    : diagnostic("Bad Options configuration") {
  }

  virtual ~CmdConfigError() throw() {
  }

  virtual const char *
  what() const throw() {
    return diagnostic.c_str();
  }

private:
  std::string diagnostic;
};


/**
 * @author hguemar <hguemar@sysfera.com>
 *
 * @class OptionNotFoundError
 * @brief Missing mandatory option
 */
class OptionNotFoundError : public CmdParserError {
public:
  explicit OptionNotFoundError(const std::string& msg)
    : diagnostic("Option not found: ") {
    diagnostic.append(msg);
  }

  virtual ~OptionNotFoundError() throw() {
  }

  virtual const char *
  what() const throw() {
    return diagnostic.c_str();
  }

private:
  std::string diagnostic;
};

/**
 * @author hguemar <hguemar@sysfera.com>
 *
 * @class OptionNoArgumentsError
 * @brief Missing option parameter
 */
class OptionNoArgumentsError : public CmdParserError {
public:
  explicit OptionNoArgumentsError(const std::string& name) {
    std::ostringstream ss(diagnostic);
    ss << "Option " << name << "has no argument.";
  }

  virtual ~OptionNoArgumentsError() throw() {
  }

  virtual const char *
  what() const throw() {
    return diagnostic.c_str();
  }

private:
  std::string diagnostic;
};


/**
 * @author hguemar <hguemar@sysfera.com>
 *
 * @class ParameterNotFoundError
 * @brief Missing parameter
 */
class ParameterNotFoundError : public CmdParserError {
public:
  explicit ParameterNotFoundError(const std::string& msg)
    : diagnostic("Missing parameter: ") {
    diagnostic.append(msg);
  }

  virtual ~ParameterNotFoundError() throw() {
  }

  virtual const char *
  what() const throw() {
    return diagnostic.c_str();
  }

private:
  std::string diagnostic;
};

#endif  /* _OPTION_EXCEPTION_HH_ */
