/**
* @file ts_set.hh
* 
* @brief  Thread safe STL vector header 
* 
* @author  Sylvain DAHAN (Sylvain.Dahan@lifc.univ-fcomte.fr)
* 
* @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/


#ifndef _TS_VECTOR_HH_
#define _TS_VECTOR_HH_

#include <cassert>
#include <memory>
#include <vector>
#include <omniconfig.h>
#include <omnithread.h>

/**
 * This is a thread safe version of the STL vector. All the methods are
 * thread safe. All the method will be added when needed. Look at the
 * STL documentation for more information.
 *
 * A vector is a Sequence that supports random access to elements,
 * constant time insertion and removal of elements at the end, and
 * linear time insertion and removal of elements at the beginning or
 * in the middle. The number of elements in a vector may vary
 * dynamically; memory management is automatic. Vector is the simplest
 * of the STL container classes, and in many cases the most efficient.
 *
 * @author Sylvain DAHAN : LIFC Besancon (France)
 */

template <class T, class A = std::allocator<T> >
class ts_vector : private std::vector<T, A> {

private:

  /**
   * This is the mutex that lock the access to the vector to avoid that
   * to thread access to the vector in the same time.
   */
  mutable omni_mutex locker;

  /**
   * A type to avoid to type vector<T, A> each time.
   */
  typedef std::vector<T, A> VectorType;

public:

  /**
   * the size_type type is the same as the vector::size_type
   */
  typedef typename VectorType::size_type size_type;

public:

  /**
   * The subscript operator performs a lookup on the key given as an
   * index and returns the corresponding value. If the key isn't
   * found, an element with the key and the default value of the
   * mapped type is inserted into the map.
   *
   * @param k index of the element of type T. k must be lesser than
   * the size of the vector.
   */
  inline T& operator[] (size_type k) {
    assert(k < size());
    locker.lock();
    T& result = VectorType::operator[](k);
    locker.unlock();
    return result;
  }

  /**
   * Resize the vector to add or remove some elements at the end of
   * the vector.
   *
   * @param n new size of the vector.
   *
   * @param t is the value of the new created element.
   */
  inline void resize(size_type n, const T& t = T()) {
    locker.lock();
    VectorType::resize(n, t);
    locker.unlock();
  }

  /**
   * return the size (number of elements) of the vector.
   */
  inline size_type size() const {
    locker.lock();
    size_type result = VectorType::size();
    locker.unlock();
    return result;
  }

  /**
   * erase all the elements of the vector. It can be a good idea to
   * make a \c clear which are thread safe before the call to the
   * destructor of the object.
   */
  inline void clear() {
    locker.lock();
    VectorType::clear();
    locker.unlock();
  }

};

#endif  // _TS_VECTOR_HH_
