/**
* @file  FIfNode.hh
* 
* @brief  The class representing the IF condition in a functional workflow
* 
* @author  Benjamin ISNARD (Benjamin.Isnard@ens-lyon.fr)
* 
* @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/


#ifndef _FIFNODE_HH_
#define _FIFNODE_HH_

#include <map>
#include <string>
#include <vector>
#include "FNode.hh"
#include "WfUtils.hh"
#include "WfExpression.hh"


/*****************************************************************************/
/*                            FIfNode class                                  */
/*****************************************************************************/


class FIfNode : public FProcNode {
public:
  FIfNode(FWorkflow* wf, const std::string& id);

  virtual ~FIfNode();

  // ******************** NODE SETUP *********************

  virtual WfPort *
  newPort(std::string portId, unsigned int ind,
          WfPort::WfPortType portType, WfCst::WfDataType dataType,
          unsigned int depth) throw(WfStructException);

  void
  setCondition(const std::string& conditionStr)
    throw(WfStructException);

  void
  setThenMap(const std::string& leftPortName, const std::string& rightPortName)
    throw(WfStructException);

  void
  setElseMap(const std::string& leftPortName, const std::string& rightPortName)
    throw(WfStructException);

  // ******************** INSTANCIATION *********************

  /**
   * Instanciation of the activity as a DagNode (inside data processing loop)
   * @param dag ref to the dag that will contain the DagNode
   * @param currTag data tag of the current dataLine
   * @param currDataLine dataLine used to set input data for the DagNode
   */
  virtual void
  createRealInstance(Dag* dag, const FDataTag& currTag,
                     std::vector<FDataHandle*>& currDataLine);

protected:
  virtual void
  checkCondition() throw(WfStructException);

  FNodePortMap  myThenMap;
  FNodePortMap  myElseMap;
  std::vector<WfExprVariable*>* myConditionVars;
  WfBooleanExpression* myCondition;
};

/*****************************************************************************/
/*                           FMergeNode class                                */
/*****************************************************************************/

class FMergeNode : public FProcNode {
public:
  FMergeNode(FWorkflow* wf, const std::string& id);

  ~FMergeNode();

  /**
   * Create a new port
   * A merge node must create 2 input ports and 1 output port to be valid
   */
  virtual WfPort *
  newPort(std::string portId, unsigned int ind,
          WfPort::WfPortType portType, WfCst::WfDataType dataType,
          unsigned int depth) throw(WfStructException);

  virtual void
  createRealInstance(Dag* dag, const FDataTag& currTag,
                     std::vector<FDataHandle*>& currDataLine);

  virtual void
  createVoidInstance(const FDataTag& currTag,
                     std::vector<FDataHandle*>& currDataLine);

private:
  void
  createMergeInstance(const FDataTag& currTag,
                      std::vector<FDataHandle*>& currDataLine);

  FNodeOutPort* myOutPort;
};

/*****************************************************************************/
/*                           FFilterNode class                               */
/*****************************************************************************/

class FFilterNode : public FProcNode {
public:
  FFilterNode(FWorkflow* wf, const std::string& id);

  virtual ~FFilterNode();

  virtual WfPort *
  newPort(std::string portId, unsigned int ind,
          WfPort::WfPortType portType, WfCst::WfDataType dataType,
          unsigned int depth) throw(WfStructException);

  virtual void
  createRealInstance(Dag* dag, const FDataTag& currTag,
                     std::vector<FDataHandle*>& currDataLine);

  virtual void
  createVoidInstance(const FDataTag& currTag,
                     std::vector<FDataHandle*>& currDataLine);

private:
  struct filterNode_t {
    FDataHandle*  dataHdl;
    bool          voidDef;
    bool          indexOk;
    bool          lastFlagOk;
    bool          isDone;
    int           newIndex;   // may be negative (-1)
    bool          newLastFlag;
  };

  void
  updateUp(FDataHandle* DH, short depth);

  void
  updateRight(const FDataTag& tag, int precIdx, short depth);

  void
  updateLeftNonVoid(const FDataTag& tag, short depth);

  void
  updateLeftVoid(const FDataTag& tag, short depth);

  void
  sendDown(FDataHandle* DH, short depth);

  // test if there is at least one non-void element (only for container)
  bool
  isNonVoid(FDataHandle* DH);

  bool
  isReadyAssumingParentIs(const FDataTag& srcTag);

  bool
  isReady(const FDataTag& srcTag);

  FDataTag
  getNewTag(const FDataTag& srcTag);

  filterNode_t*
  getTreeNode(FDataHandle* DH);

  filterNode_t*
  getTreeNode(const FDataTag& tag);

  FNodeOutPort* myOutPort;

  std::map<FDataTag, filterNode_t*> myTree;
};


#endif  // _FIFNODE_HH_
