/**
 * @file DIETSecurityManager.cc
 *
 * @brief  Utility class for handling security for DIET
 *
 * @author Guillaume Verger (guillaume.verger@inria.fr)
 *
 * @section License
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

// CMake variables
#include "security_config.h"

#ifdef DIET_USE_SECURITY
#include "DIETSecurityManager.hh"

#include <string>
#include <set>
#include <vector>
#include <boost/program_options.hpp>
#include <boost/foreach.hpp>

#include <omniORB4/CORBA.h>
#include <omniORB4/sslContext.h>

#include "debug.hh"
#include "configuration.hh"

namespace po = boost::program_options;

static const std::string UNDEF = "UNDEFINED";
static const std::string EMPTY = "";

DIETSecurityManager::DIETSecurityManager() : enabled(false), cAFile(UNDEF), keyFile(UNDEF), passwordKey(EMPTY) {
}

bool
DIETSecurityManager::initSSLContext() {
  /*
   *  Setting SSL Contex
   */
  CONFIG_BOOL(diet::SSLENABLED, this->enabled);

  if (this->enabled) {

    CONFIG_STRING(diet::SSLROOTCERTIFICATE, this->cAFile);
    CONFIG_STRING(diet::SSLPRIVATEKEY, this->keyFile);
    CONFIG_STRING(diet::SSLPRIVATEKEYPASSWORD, this->passwordKey);

    sslContext::certificate_authority_file = cAFile.c_str();
    sslContext::key_file = keyFile.c_str();
    sslContext::key_file_password = passwordKey.c_str();

    sslContext::verify_mode = SSL_VERIFY_FAIL_IF_NO_PEER_CERT | SSL_VERIFY_PEER;
  }
  else {
    sslContext::verify_mode = SSL_VERIFY_NONE;
  }
  return true;
}


DIETSecurityManager::~DIETSecurityManager() {
  if (this->enabled) {
    BOOST_FOREACH(char * opt, secuOptions) {
      delete[] opt;
    }
  }
}

char *
strToCharPtr(const std::string &str) {
  char * result = new char[str.size()+1];
  strcpy(result, str.c_str());
  return result;
}

void
insertOptions(const std::string &option, const std::vector<std::string> &values, std::vector< char * > &toFill) {
  BOOST_FOREACH(std::string val, values) {
    toFill.push_back(strToCharPtr(option));
    toFill.push_back(strToCharPtr(val));
  }
}

void
insertOptions(const std::string &option, const std::set<std::string> &values, std::vector< char * > &toFill) {
  BOOST_FOREACH(std::string val, values) {
    toFill.push_back(strToCharPtr(option));
    toFill.push_back(strToCharPtr(val));
  }
}


bool
DIETSecurityManager::secureORBOptions(int argc, char * argv[]) {


  if (!this-> enabled) {

    for (int i = 0; i < argc; ++i) {
      this->secuOptions.push_back(argv[i]);
    }
    return true;
  }


  std::vector<std::string> oEndPoint;
  //  std::vector<std::string> oEndPointPublish;
  std::vector<std::string> oServerTransportRule;
  std::vector<std::string> oClientTransportRule;

  po::options_description desc("Handled Options");
  desc.add_options()
      ("ORBendPoint", po::value<std::vector<std::string> >(&oEndPoint), "end point description")
      // ("ORBendPointPublish", po::value<std::vector <std::string> >(&oEndPointPublish), "end point publishing description")
      ("ORBserverTransportRule", po::value<std::vector<std::string> >(&oServerTransportRule), "server transport rule")
      ("ORBclientTransportRule", po::value<std::vector<std::string> >(&oClientTransportRule), "client transport rule")
      ;

  po::variables_map vm;

  try {
    po::parsed_options opts = po::command_line_parser(argc, argv).style(
        po::command_line_style::allow_long_disguise
            | po::command_line_style::long_allow_next).allow_unregistered().options(desc).run();


    std::vector<std::string> unmodifiedOptions = po::collect_unrecognized(opts.options, po::include_positional);
    BOOST_FOREACH(std::string uOpt, unmodifiedOptions) {
      secuOptions.push_back(strToCharPtr(uOpt));
      TRACE_TEXT(TRACE_MAIN_STEPS, "unmodified option : " << uOpt << std::endl);
    }

    po::store(opts, vm);
    po::notify(vm);

    // Securing end points
    std::set<std::string> endPointToSet;
    if (oEndPoint.empty()) {
      endPointToSet.insert("giop:tcp::");
      endPointToSet.insert("giop:ssl::");
      TRACE_TEXT(TRACE_MAIN_STEPS, "Changing end point configuration : allowing ssl and tcp endpoints." << std::endl);
    } else {
      BOOST_FOREACH(std::string endPoint, oEndPoint) {
        std::vector<std::string>::size_type idx = endPoint.find(":tcp:");
        if (idx != std::string::npos) {
          std::string secuEP = std::string(endPoint).replace(idx, 5, ":ssl:");
          endPointToSet.insert(secuEP);
        }
        endPointToSet.insert(endPoint);
      }
    }

    // Filling end point options
    insertOptions("-ORBendPoint", endPointToSet, secuOptions);


    // Securing server rules: only accept incoming ssl requests
    std::vector<std::string> serverRuleToSet;

    if (!oServerTransportRule.empty()) {
      BOOST_FOREACH(std::string serverTR, oServerTransportRule) {
        std::vector<std::string>::size_type idx = serverTR.find("tcp");
        if (idx == std::string::npos) {
          serverRuleToSet.push_back(serverTR);
        } else {
          WARNING("Ignoring server transport rule [" << serverTR
              << "] : tcp is not allowed !" << std::endl);
        }
      }
    }
    if (serverRuleToSet.empty()) {
      // Test with SSL Forwarder
      serverRuleToSet.push_back("localhost unix,tcp,ssl");
      serverRuleToSet.push_back("* ssl");
    }
    // Filling server rule options
    insertOptions("-ORBserverTransportRule", serverRuleToSet, secuOptions);


    // Securing client rules: prefer sending ssl requests
    std::vector<std::string> clientRuleToSet;
    if (!oClientTransportRule.empty()) {
      BOOST_FOREACH(std::string clientTR, oClientTransportRule) {
        std::vector<std::string>::size_type idxTCP = clientTR.find("tcp");
        std::string rule = clientTR;
        if (idxTCP != std::string::npos) {
          std::vector<std::string>::size_type idxSSL = clientTR.find("ssl");
          if (idxSSL == std::string::npos || idxSSL > idxTCP) {
            rule.replace(idxTCP, 3, "ssl");
            if (idxSSL > idxTCP) {
              rule.replace(idxSSL, 3, "tcp");
            }
            TRACE_TEXT(
                TRACE_ALL_STEPS,
                "Rewriting client transport rule [" << clientTR << "] into [" <<
                rule << "] to fit security needs." << std::endl);

          }
        }
        clientRuleToSet.push_back(rule);
      }
    }
    if (clientRuleToSet.empty()) {
      // Test with SSL Forwarder
      clientRuleToSet.push_back("localhost unix,tcp,ssl");
      clientRuleToSet.push_back("* unix,ssl,tcp");
    }

    // Filling server rule options
    insertOptions("-ORBclientTransportRule", clientRuleToSet, secuOptions);

  } catch (const std::exception &e) {
    WARNING("Exception raised while processing security options for the Orb : " << e.what() << std::endl);
    return false;
  }
  return true;
}

bool DIETSecurityManager::enableSecurity(int argc, char * argv[]) {
  this->initSSLContext();
  this->secureORBOptions(argc, argv);

  return true;
}

#endif
