/**
 * @file Counter.cc
 *
 * @brief  Thread safe counter implementation
 *
 * @author  Sylvain DAHAN (Sylvain.Dahan@lifc.univ-fcomte.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "Counter.hh"
#include <cassert>


Counter::Counter(const Counter &aCounter) {
  value = static_cast<CORBA::ULong>(aCounter);
}

Counter
Counter::operator++(int i) {
  valueMutex.lock();
  assert(value < value + 1);  // check for overflow
  Counter oldValue;
  oldValue.value = value;
  value++;
  valueMutex.unlock();
  return oldValue;
}

Counter
Counter::operator--(int i) {
  valueMutex.lock();
  assert(value > 0);
  Counter oldValue;
  oldValue.value = value;
  value--;
  valueMutex.unlock();
  return oldValue;
}

Counter &
Counter::operator=(const Counter &aCounter) {
  value = static_cast<CORBA::ULong>(aCounter);
  return *this;
}


Counter::operator CORBA::ULong() const {
  CORBA::ULong valueBuf;
  valueMutex.lock();
  valueBuf = value;
  valueMutex.unlock();
  return valueBuf;
}

Counter &
Counter::operator--() {
  assert(value > 0);
  valueMutex.lock();
  value--;
  valueMutex.unlock();
  return *this;
}

Counter &
Counter::operator++() {
  assert(value < value + 1);
  valueMutex.lock();
  value++;
  valueMutex.unlock();
  return *this;
}

Counter &
Counter::operator-=(const Counter &aCounter) {
  assert(value > 0);
  valueMutex.lock();
  value = value - static_cast<CORBA::ULong>(aCounter);
  valueMutex.unlock();
  return *this;
}

Counter &
Counter::operator+=(const Counter &aCounter) {
  assert(value < value + 1);
  valueMutex.lock();
  value = value + static_cast<CORBA::ULong>(aCounter);
  valueMutex.unlock();
  return *this;
}
