/**
 * @file progs.h
 *
 * @brief  dmat_manips example: matrix manipulations programs
 *
 * @author  Philippe COMBES (Philippe.Combes@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef _PROGS_H_
#define _PROGS_H_

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef WIN32
#define INLINE __inline
#else 
#define INLINE inline
#endif


#define print_matrix(mat, m, n, rm) {           \
    size_t i, j;                                \
    printf("%s (%s-major) = \n", # mat,          \
           (rm) ? "row" : "column");            \
    for (i = 0; i < (m); i++) {                 \
      for (j = 0; j < (n); j++) {               \
        if (rm) {                               \
          printf("%3f ", (mat)[j + i * (n)]);     \
        } else {                                \
          printf("%3f ", (mat)[i + j * (m)]);     \
        }                                       \
      }                                         \
      printf("\n");                             \
    }                                           \
    printf("\n");                               \
}

/*
 * Transpose a matrix (column-major <=> rm == 0)
 */

INLINE int
T(int m, int n, double *A, int rm) {
  size_t i, j;
  double *tmp = NULL;

  tmp = malloc(m * n * sizeof(tmp));
  memcpy(tmp, A, m * n * sizeof(tmp));

  for (i = 0; i < n; i++) {
    for (j = 0; j < m; j++) {
      if (rm) {
        A[i * m + j] = tmp[j * n + i];
      } else {
        A[j * n + i] = tmp[i * m + j];
      }
    }
  }

  free(tmp);
  return 0;
} // T

/*
 * Sum 2 column-major matrices (modulo tA and tB):
 * if tA == 'T', then A is row-major ...
 */

INLINE int
MatSUM(char tA, char tB, int m, int n, double *A, double *B, double *C) {
  size_t i, j;

  if (tA == 'T') {
    if (tB == 'T') {
      for (i = 0; i < m; i++) {
        for (j = 0; j < n; j++) {
          C[j * m + i] = A[i * n + j] + B[i * n + j];
        }
      }
    } else {
      for (i = 0; i < m; i++) {
        for (j = 0; j < n; j++) {
          C[j * m + i] = A[i * n + j] + B[j * m + i];
        }
      }
    }
  } else {
    if (tB == 'T') {
      for (i = 0; i < m; i++) {
        for (j = 0; j < n; j++) {
          C[j * m + i] = A[j * m + i] + B[i * n + j];
        }
      }
    } else {
      for (i = 0; i < m; i++) {
        for (j = 0; j < n; j++) {
          C[j * m + i] = A[j * m + i] + B[j * m + i];
        }
      }
    }
  }

  return 0;
} // MatSUM


/*
 * Multiply 2 column-major matrices (modulo tA and tB):
 * if tA == 'T', then A is row-major ...
 */

INLINE int
MatPROD(char tA, char tB,
        int mA, int nA, double *A,
        int nB, double *B, double *C) {
  size_t i, j, k;
  int mB = nA;

  if (tA == 'T') {
    if (tB == 'T') {
      for (i = 0; i < mA; i++) {
        for (j = 0; j < nB; j++) {
          C[j * mA + i] = 0;
          for (k = 0; k < nA; k++) {
            C[j * mA + i] += A[i * nA + k] * B[k * nB + j];
          }
        }
      }
    } else {
      for (i = 0; i < mA; i++) {
        for (j = 0; j < nB; j++) {
          C[j * mA + i] = 0;
          for (k = 0; k < nA; k++) {
            C[j * mA + i] += A[i * nA + k] * B[j * mB + k];
          }
        }
      }
    }
  } else {
    if (tB == 'T') {
      for (i = 0; i < mA; i++) {
        for (j = 0; j < nB; j++) {
          C[j * mA + i] = 0;
          for (k = 0; k < nA; k++) {
            C[j * mA + i] += A[k * mA + i] * B[k * nB + j];
          }
        }
      }
    } else {
      for (i = 0; i < mA; i++) {
        for (j = 0; j < nB; j++) {
          C[j * mA + i] = 0;
          for (k = 0; k < nA; k++) {
            C[j * mA + i] += A[k * mA + i] * B[j * mB + k];
          }
        }
      }
    }
  }

  return 0;
} // MatPROD

#endif  // _PROGS_H_
