/**
 * @file Transfers.cc
 *
 * @brief  Object to obtain transfers progression.
 *
 * @author Gael Le Mahec (lemahec@clermont.in2p3.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include <string>
#include <map>
#include <stdexcept>
#include <iostream>
#include "Transfers.hh"
#include "DagdaFactory.hh"

Transfers *Transfers::instance = NULL;

Transfers *
Transfers::getInstance() {
  if (instance == NULL) {
    instance = new Transfers(DagdaFactory::getMaxMsgSize());
  }
  return instance;
}

void
Transfers::newTransfer(const std::string &id,
                       const unsigned long long total) {
  progress[id].total = total;
  progress[id].transfered = 0;
}


void
Transfers::incProgress(const std::string &id) {
  std::map<std::string, struct transfer_t>::iterator it;

  if ((it = progress.find(id)) != progress.end()) {
    it->second.transfered += msgSize;
    if (it->second.transfered > it->second.total) {
      it->second.transfered = it->second.total;
    }
  } else {
    throw std::runtime_error("Unknown transfer: " + id);
  }
} // incProgress

Transfers::Transfers(unsigned long long msgSize) {
  this->msgSize = msgSize;
}

unsigned long long
Transfers::totalSize(const std::string &id) const {
  std::map<std::string, struct transfer_t>::const_iterator it;

  if ((it = progress.find(id)) != progress.end()) {
    return it->second.total;
  }
  throw std::runtime_error("Unknown transfer: " + id);
}

unsigned long long
Transfers::getTransfered(const std::string &id) const {
  std::map<std::string, struct transfer_t>::const_iterator it;

  if ((it = progress.find(id)) != progress.end()) {
    return it->second.transfered;
  }
  throw std::runtime_error("Unknown transfer: " + id);
}

double
Transfers::getProgress(const std::string &id) {
  return ((double) getTransfered(id)) / ((double) totalSize(id));
}

void
Transfers::remTransfer(const std::string &id) {
  std::map<std::string, struct transfer_t>::iterator it;

  if ((it = progress.find(id)) != progress.end()) {
    progress.erase(it);
  }
  throw std::runtime_error("Unknown transfer: " + id);
}
