/**
 * @file  JobQueue.hh
 *
 * @brief  Container for SeD jobs status information
 *
 * @author  Benjamin ISNARD (benjamin.isnard@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef _JOBQUEUE_HH_
#define _JOBQUEUE_HH_

#include <ctime>
#include <list>
#include <map>
#include <omnithread.h>
#include "DIET_data_internal.hh"
#include "debug.hh"

class JobQueue {
public:
  explicit
  JobQueue(int maxConcurrentJobsNb = 1);

  ~JobQueue();

  /**
   * Add a job in the waiting queue for the SeD
   * (called when the SOLVE request arrives)
   * @param dietReqID the identifier of the request
   * @param jobEstCompTime  the estimated duration of the job
   * @param ev  the vector of estimations (for bw compatibility)
   */
  void
  addJobWaiting(int dietReqID, double jobEstCompTime, corba_estimation_t &ev);

  /**
   * Change the status of the job to RUNNING
   * (called when the SOLVE request gets a ressource)
   * @param dietReqId the identifier of the request
   */
  bool
  setJobStarted(int dietReqId);

  /**
   * Change the status of the job to FINISHED
   * @param dietReqId the identifier of the request
   */
  bool
  setJobFinished(int dietReqId);

  bool
  deleteJob(int dietReqId);

  /**
   * @deprecated (kept for backward compatibility)
   * Get the stored list of jobs
   * @param jobVector ref to job table
   */
  int
  getActiveJobTable(jobVector_t &jobVector);

  /**
   * Earliest Finish Time estimation using FIFO scheduling
   * This method consider all running and waiting jobs in the queue
   * and calculates the EFT using maxConcurrentJobsNb processors.
   * @return  the EFT value from now (in ms)
   */
  double
  estimateEFTwithFIFOSched();

private:
  // / Container for the list of jobs
  std::map<int, diet_job_t>  myJobs;

  // / FIFO queue to store waiting jobs (for EFT processing)
  std::list<int>   myWaitingQueue;

  // / Lock for thread-safe access to variables
  omni_mutex myLock;

  // / Counter of active jobs (ie not finished)
  int nbActiveJobs;

  // / Number of processors
  int nbProc;
};

#endif  // _JOBQUEUE_HH_
