/**
 * @file  AccessController.hh
 *
 * @brief  Interface for limiting access to a resource to a configurable consumers
 *
 * @author  Holly DAIL (Holly.Dail@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef _ACCESSCONTROLLER_HH_
#define _ACCESSCONTROLLER_HH_

#include <cassert>
#include <sys/types.h>
#include <omniORB4/CORBA.h>
#include <omnithread.h>


class AccessController {
public:
  /**
   * Creates a new AccessController with a resource limit of \c
   * initialResources.
   *
   * @param initialResources The number of resources initially available
   *                  It must be a non-negative integer.
   */
  explicit
  AccessController(int initialResources);

  void
  waitForResource();

  void
  releaseResource();

  /**
   * Get count of resources immediately available.
   */
  int
  getFreeSlotCount();

  /**
   * Get count of number of threads waiting for a resource.
   */
  int
  getNumWaiting();

  /**
   * Get number of available resources.
   */
  int
  getTotalResourceCount();

private:
  // / The number of resources initially available.
  int resourcesInitial;

  // / Lock for thread-safe access to variables
  omni_mutex globalLock;

  // / Semaphore to provide low-level resource limit
  omni_semaphore *resourceSemaphore;

  /** Count of number of free resource slots.
   * [Needed because omni_semaphore will not report semaphore value. */
  int numFreeSlots;

  // / Number of threads waiting for resource access
  int numWaiting;

  // / Assign unique req ID to each access request
  int reqCounter;

  // / Largest ID of request that has been given resource
  int maxIDReleased;
};

#endif  // _ACCESSCONTROLLER_HH_
