/**
 * @file  CltDagNodeLauncher.cc
 *
 * @brief   Launcher for client-side dag node
 *
 * @author   Benjamin ISNARD (benjamin.isnard@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "CltDagNodeLauncher.hh"
#include "debug.hh"
#include "DIETCall.hh"
#include "Dag.hh"
#include "DagNode.hh"
#include "EventTypes.hh"
#include "ORBMgr.hh"

using namespace events;

#define MAX_EXEC_SERVERS 10 \
  // max nb of servers in the agent response

CltDagNodeLauncher::CltDagNodeLauncher(DagNode *parent)
  : DagNodeLauncher(parent) {
}

std::string
CltDagNodeLauncher::toString() const {
  return "Client " + DagNodeLauncher::toString();
}


void
CltDagNodeLauncher::execNode() {
  std::string traceHeader = "[" + myNode->getId() + "] Client Launcher : ";
  TRACE_TEXT(TRACE_ALL_STEPS, traceHeader << "starting... \n");

  sendEventFrom<DagNodeLauncher,
                DagNode::START>(this,
                                "Start client launcher",
                                "",
                                EventBase::INFO);

  // set the reqId in the profile
  if (isSeDDefinedFlag) {
    myNode->getProfile()->dietReqID = (int) myReqID;
    TRACE_TEXT(TRACE_ALL_STEPS, traceHeader << "setting reqID in profile to #"
                                            << myReqID << "\n");
  }

  // diet call
  try {
    SeD_var sed;
    if (myChosenServer == NULL) {
      sed = SeD::_nil();
    } else {
      sed = ORBMgr::getMgr()->resolve<SeD, SeD_var>(SEDCTXT, myChosenServer);
    }

    if (!diet_call_common(myNode->getDag()->getExecutionAgent(),
                          myNode->getProfile(),
                          sed,
                          &myEstimVect,
                          MAX_EXEC_SERVERS)) {
      isSuccessfulFlag = true;
      TRACE_TEXT(TRACE_MAIN_STEPS, traceHeader << "diet call DONE reqID=" <<
                 myNode->getProfile()->dietReqID << "\n");
      myNode->storeProfileData();
    } else {
      WARNING(traceHeader << "diet call FAILED\n");
      isSuccessfulFlag = false;
    }
  } catch (Dagda::DataNotFound &e) {
    WARNING(traceHeader << "Data not found (ID=" << e.dataID << ")");
    isSuccessfulFlag = false;
  } catch (Dagda::ReadError &e) {
    WARNING(traceHeader << "Data read error");
    isSuccessfulFlag = false;
  } catch (Dagda::InvalidPathName &e) {
    WARNING(traceHeader << "Invalid path name");
    isSuccessfulFlag = false;
  } catch (Dagda::UnreachableFile &e) {
    WARNING(traceHeader << "Unreachable file");
    isSuccessfulFlag = false;
  } catch (Dagda::PermissionDenied &e) {
    WARNING(traceHeader << "Permission denied");
    isSuccessfulFlag = false;
  } catch (Dagda::UnavailableData &e) {
    WARNING(traceHeader << "Unavailable data");
    isSuccessfulFlag = false;
  } catch (CORBA::SystemException &e) {
    WARNING(traceHeader << "Got a CORBA " << e._name() << " exception ("
                        << e.NP_minorString() << ")");
    isSuccessfulFlag = false;
  }
} // execNode

void
CltDagNodeLauncher::finishNode() {
  if (isSuccessfulFlag) {
    sendEventFrom<DagNodeLauncher, DagNode::FINISH>(this,
                                                    "Node exec successful", "",
                                                    EventBase::INFO);
  } else {
    sendEventFrom<DagNodeLauncher, DagNode::FAILED>(this, "Node exec failed",
                                                    "",
                                                    EventBase::NOTICE);
  }
}
