/**
 * @file CacheAlgorithms.cc
 *
 * @brief  Cache replacements algorithms used by DAGDA when we remove data
 *
 * @author  Gael Le Mahec (lemahec@clermont.in2p3.fr)
 *
 * @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

#include <iostream>
#include <string>
#include <map>

#include "AdvancedDagdaComponent.hh"
#include "DIET_data_internal.hh"
#include "debug.hh"

#ifdef WIN32
   #define DIET_API_LIB __declspec(dllexport)
#else
   #define DIET_API_LIB
#endif
using namespace std;

extern "C" DIET_API_LIB int
LRU(AdvancedDagdaComponent *manager, size_t size, dagda_object_type_t type) {
  TRACE_TEXT(TRACE_ALL_STEPS, "Needs more space for the data:" <<
             " Tries to remove one using LRU." << endl);
  std::map<std::string, corba_data_t>::iterator it;
  clock_t leastRecent = 0;
  std::string found;

  for (it = manager->getData()->begin(); it != manager->getData()->end(); ++it)
    if (it->second.desc.mode != DIET_STICKY && DGD_OBJ_TYPE(it->second) ==
        type) {
      if ((leastRecent == 0 || leastRecent >
           manager->getLastUsageTime(it->first.c_str()))
          && data_sizeof(&it->second.desc) >= size) {
        leastRecent = manager->getLastUsageTime(it->first.c_str());
        found = it->first;
      }
    }
  if (found != "") {
    manager->remData(found.c_str());
    return 0;
  } else {cerr << "No sufficient space found..." << endl;
  }
  return 1;
} // LRU

int
LFU(AdvancedDagdaComponent *manager, size_t size, dagda_object_type_t type) {
  TRACE_TEXT(TRACE_ALL_STEPS, "Needs more space for the data:" <<
             " Tries to remove one using LFU policy." << endl);
  std::map<std::string, corba_data_t>::iterator it;
  unsigned long usageMin = 0;
  std::string found;

  for (it = manager->getData()->begin(); it != manager->getData()->end(); ++it)
    if (it->second.desc.mode != DIET_STICKY && DGD_OBJ_TYPE(it->second) ==
        type) {
      if ((usageMin == 0 || usageMin > manager->getNbUsage(it->first.c_str()))
          && data_sizeof(&it->second.desc) >= size) {
        usageMin = manager->getNbUsage(it->first.c_str());
        found = it->first;
      }
    }
  if (found != "") {
    manager->remData(found.c_str());
    return 0;
  } else {cerr << "No sufficient space found..." << endl;
  }
  return 1;
} // LFU

int
FIFO(AdvancedDagdaComponent *manager, size_t size, dagda_object_type_t type) {
  TRACE_TEXT(TRACE_ALL_STEPS, "Needs more space for the data:" <<
             " Tries to remove one using FIFO policy." << endl);
  std::map<std::string, corba_data_t>::iterator it;
  clock_t registerTime = 0;
  std::string found;

  for (it = manager->getData()->begin(); it != manager->getData()->end(); ++it)
    if (it->second.desc.mode != DIET_STICKY && DGD_OBJ_TYPE(it->second) ==
        type) {
      if ((registerTime == 0 || registerTime >
           manager->getRegisterTime(it->first.c_str()))
          && data_sizeof(&it->second.desc) >= size) {
        registerTime = manager->getRegisterTime(it->first.c_str());
        found = it->first;
      }
    }
  if (found != "") {
    manager->remData(found.c_str());
    return 0;
  } else {cerr << "No sufficient space found..." << endl;
  }
  return 1;
} // FIFO
