/**
* @file FActivityNode.hh
* 
* @brief  The nodes of a functional workflow that correspond to tasks in the DAG 
* 
* @author   Benjamin ISNARD (Benjamin.Isnard@ens-lyon.fr)
* 
* @section Licence
 *
 * Copyright ENS Lyon, INRIA, UCBL, SysFera (2000)
 *
 * - Frederic.Desprez@ens-lyon.fr (Project Manager)
 * - Eddy.Caron@ens-lyon.fr (Technical Manager)
 * - Tech@sysfera.com (Maintainer and Technical Support)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/

#ifndef _FACTIVITYNODE_HH_
#define _FACTIVITYNODE_HH_

#include <string>
#include <vector>
#include "FNode.hh"

class FActivityNode : public FProcNode {
public:
  FActivityNode(FWorkflow* wf, const std::string& id);

  virtual ~FActivityNode();

  /**
   * Set the maximum nb of instances of this node inside one dag
   * @param maxInst the max nb of instances
   */
  void
  setMaxInstancePerDag(short maxInst);

  /**
   * Set DIET service name
   * @param path the DIET service name
   */
  void
  setDIETServicePath(const std::string& path);

  /**
   * Set DIET estimation option
   * If estimOption == "constant" the instanciator will use the
   * 'est-class' attribute for all generated dag nodes
   * @param estimOption = ('constant'|...)
   */
  void
  setDIETEstimationOption(const std::string& estimOption);

  /**
   * Initialization
   *  - connection to other nodes
   *  - input iterators setup
   */
  virtual void
  initialize();

  /**
   * Specific action to take before starting data processing loop
   */
  virtual void
  initInstanciation();

  /**
   * Test if the instanciation can continue
   * @return true if instanciation must stop
   */
  virtual bool
  instLimitReached();

  /**
   * Instanciation of the activity as a DagNode (inside data processing loop)
   * @param dag ref to the dag that will contain the DagNode
   * @param currTag data tag of the current dataLine
   * @param currDataLine dataLine used to set input data for the DagNode
   */
  virtual void
  createRealInstance(Dag* dag, const FDataTag& currTag,
                     std::vector<FDataHandle*>& currDataLine);

  /**
   * Update the FNode instanciation status after data processing loop
   */
  virtual void
  updateInstanciationStatus();

protected:
  /**
   * The service path
   */
  std::string myPath;

  /**
   * The estimation option
   */
  std::string myEstimOption;

  /**
   * Max number of instances per dag
   */
  short maxInstNb;

private:
  /**
   * Nb of instances created in current dag
   */
  int nbInstances;
};

#endif  // _FACTIVITYNODE_HH_






