/**
 * @file MaDagNodeLauncher.cc
 *
 * @brief  The specialized launcher class used for dagNode scheduling
 *
 * @author  Benjamin Isnard (Benjamin.Isnard@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "MaDagNodeLauncher.hh"
#include "debug.hh"
#include "Dag.hh"
#include "DagNode.hh"

using namespace madag;

MaDagNodeLauncher::MaDagNodeLauncher(DagNode *parent,
                                     MultiWfScheduler *scheduler,
                                     CltMan_ptr clientMgr)
  : DagNodeLauncher(parent, scheduler), myScheduler(scheduler),
  myCltMgr(clientMgr) {
}

void
MaDagNodeLauncher::execNode() {
  Dag *dag = myNode->getDag();
  std::string dagId = dag->getId();
  std::string nodeId = myNode->getId();
  std::string nodePx = "[" + myNode->getCompleteId() + "] : ";
  bool clientFailure = false;
  CORBA::Long res;

  try {
    if (isSeDDefinedFlag) {
      TRACE_TEXT(
        TRACE_ALL_STEPS, nodePx << "call client (sed defined) - request #"
                                << this->myReqID << "\n");
      res = myCltMgr->execNodeOnSed(nodeId.c_str(),
                                    dagId.c_str(),
                                    myChosenServer,
                                    (CORBA::ULong) myReqID,
                                    myEstimVect);
    } else {
      TRACE_TEXT(TRACE_ALL_STEPS, nodePx << "call client (sed not defined)\n");
      res = myCltMgr->execNode(nodeId.c_str(),
                               dagId.c_str());
    }
    TRACE_TEXT(TRACE_MAIN_STEPS, nodePx << "call client DONE\n");
  } catch (CORBA::COMM_FAILURE &e) {
    WARNING(nodePx << "Client call had connection problems\n");
    clientFailure = true;
  } catch (CORBA::SystemException &e) {
    WARNING(nodePx << "Client call got a CORBA " << e._name() << " exception ("
                   << e.NP_minorString() << ")\n");
    clientFailure = true;
  } catch (...) {
    WARNING(nodePx << "Client call got unknown exception!\n");
    clientFailure = true;
  }

  isSuccessfulFlag = !clientFailure && !res;
} // execNode

void
MaDagNodeLauncher::finishNode() {
  myScheduler->wakeUp(false, myNode);  // scheduler main thread will join my thread
}
