#ifndef __GENERICSIMPLEWFCLIENTTEST_HPP__
#define __GENERICSIMPLEWFCLIENTTEST_HPP__


#include <DIET_data.h>


void
genericSimpleWorkflowClient(const std::string &wfFileName,
                            const std::string &wfName,
                            const wf_level_t &wfType,
                            const std::string &dataFileName = "",
                            const std::string &dataOutFileName = "data_out.xml",
                            const std::string &transcriptFileName = "",
							const std::string &expectedResult = "") {

  BOOST_TEST_MESSAGE("Generic Workflow client.\nParameters:");
  BOOST_TEST_MESSAGE("Workflow filename: " << wfFileName);
  BOOST_TEST_MESSAGE("Workflow name: " << wfName);
  if (wfType == DIET_WF_FUNCTIONAL) {
    BOOST_TEST_MESSAGE("Workflow type: DIET_WF_FUNCTIONAL");
  } else {
    BOOST_TEST_MESSAGE("Workflow type: DIET_WF_DAG");
  }
  BOOST_TEST_MESSAGE("Data filename: " << dataFileName);
  BOOST_TEST_MESSAGE("Data out filename: " << dataOutFileName);
  BOOST_TEST_MESSAGE("Transcript filename: " << transcriptFileName);
  BOOST_TEST_MESSAGE("Expected result: " << expectedResult);

  utils::ClientArgs c("files", "client_testing_wf.cfg");

  diet_error_t error = diet_initialize(c.config(), c.argc(), c.argv());

  // check if diet_initialize don't return any error
  BOOST_REQUIRE_MESSAGE(GRPC_NO_ERROR == error,
                        "diet_initialize() should return "
                        << diet_error_string(GRPC_NO_ERROR)
                        << " instead of "
                        << diet_error_string(error));


  /*
   * Allocate the workflow profile
   */
  diet_wf_desc_t *profile;
  profile = diet_wf_profile_alloc(wfFileName.c_str(), wfName.c_str(), wfType);

  BOOST_CHECK(profile != NULL);
  if (profile == NULL){
	  //Useless to keep go but we still need to call diet_finalize so that
	  //we don't impact the other following tests
	  diet_finalize();
	  return;
  }


  if (wfType == DIET_WF_FUNCTIONAL) {
    /*
     * For functional workflows ONLY
     * Defines which file is used to provide the data to instanciate the wf
     */
    diet_wf_set_data_file(profile, dataFileName.c_str());

    /*
     * For workflow restart
     * Defines which file is used to store the execution transcriptFileName
     * (file will be overwritten if existing)
     */
    if (!transcriptFileName.empty()) {
      diet_wf_set_transcript_file(profile, transcriptFileName.c_str());
    }
  }


  /* Execute workflow */
  error = diet_wf_call(profile);
  BOOST_CHECK_EQUAL(GRPC_NO_ERROR, error);

  /* Save output data */
  if (wfType == DIET_WF_FUNCTIONAL) {
    error = diet_wf_save_data_file(profile, dataOutFileName.c_str());
    BOOST_CHECK_EQUAL(GRPC_NO_ERROR, error);
  }

  /* Display result */
  std::string obtainedResult;
  if (expectedResult.empty()){
	  BOOST_TEST_MESSAGE("Not checking the actual result of test " << wfName);
	  error = diet_wf_print_results(profile, "");
  } else {

	  //Ugly and unsafe but the normal solution with ifstream doesn't compile
	  //on my computer
	  std::string resultfile = WF_DIRECTORY "result_wf.txt";
	  error = diet_wf_print_results(profile, resultfile.c_str());

	  /* Checking the actual result against what is expected for the test */
	  FILE *results = fopen(resultfile.c_str(), "r");
	  if (results != NULL){
		 char line [1024];
		 while (fgets(line, 1024, results) != NULL){
			obtainedResult.append(line);
		 }

		fclose(results);
	  }

	  std::cerr << "Expected WF result was [ " << expectedResult
			  << " ] \n but we got [ " << obtainedResult
			  << " ] instead ";

  }


  BOOST_CHECK_EQUAL(GRPC_NO_ERROR, error);


  if (!transcriptFileName.empty() && wfType == DIET_WF_FUNCTIONAL) {
    error = diet_wf_save_transcript_file(profile, transcriptFileName.c_str());
    BOOST_CHECK_EQUAL(GRPC_NO_ERROR, error);
  }



  //diet_wf_profile_free(profile);
  diet_wf_free(profile);

  // now we're good
  BOOST_TEST_MESSAGE("-- Now calling diet_finalize ");
  error = diet_finalize();
  BOOST_REQUIRE_MESSAGE(GRPC_NO_ERROR == error,
                        "diet_finalize() should return "
                        << diet_error_string(GRPC_NO_ERROR)
                        << " instead of "
                        << diet_error_string(error));

  //Checking the result at the end so that we are sure diet_finalize() is called
  BOOST_REQUIRE_MESSAGE(expectedResult == obtainedResult,
  			  "Expected WF result was [ " << expectedResult
  			  << " ] \n but we got [ " << obtainedResult
  			  << " ] instead ");



} // genericSimpleWorkflowClient


#endif // ifndef __GENERICSIMPLEWFCLIENTTEST_HPP__
