/**
 * @file Deltacloud_BatchSystem.cc
 *
 * @brief Implementation of the Deltacloud batchsystem
 *
 * @author Guillaume Verger (guillaume.verger@inria.fr)
 *         Lamiel Toch (lamiel.toch@ens-lyon.fr)
 *
 * @section License
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

#include "Deltacloud_BatchSystem.hh"

#include "DIET_server.h"
#include "DIET_uuid.hh"
#include "debug.hh"
#include "configuration.hh"

#include <boost/filesystem.hpp>

#include <fstream>
#include <sstream>

#include <stdlib.h> // free, malloc
#include <string.h> // strdup

#define NB_TRIES_MAX 10 // Number of times we try to access to vm before fail
#define CONNECT_TIMEOUT 10 // Number of seconds to consider ssh cannot connect

#define DC_PRINT(dcfun) #dcfun
#define ENSURE(dcfunction,onfail) if (dcfunction < 0) {\
 WARNING("Deltacloud error at " << DC_PRINT(dcfunction) << " : " <<\
      deltacloud_get_last_error_string());\
  onfail;\
} //else {\
  std::cout << "Done : " << DC_PRINT(dcfunction) << std::endl;}

Deltacloud_BatchSystem::Deltacloud_BatchSystem(
    int batchID, const char *batchName) {

  this->batchName = batchName;
  this->batch_ID = batchID;

  memset(&api, 0, sizeof(struct deltacloud_api));
  init();

}

Deltacloud_BatchSystem::~Deltacloud_BatchSystem () {
}

bool
copyTo(char * address, std::string file, std::string out_dir) {
      std::stringstream ss;
      ss << "scp -o StrictHostKeyChecking=no -o ConnectTimeOut=" <<
       CONNECT_TIMEOUT << " " << file << " root@" << address << ":" << out_dir;
      return system(ss.str().c_str()) == 0;
}

bool
runSsh(std::string address, std::string command) {
      std::stringstream ss;
      ss << "ssh -o StrictHostKeyChecking=no -o ConnectTimeOut=" <<
        CONNECT_TIMEOUT << " " << "root@" << address << " " << command;
      return system(ss.str().c_str()) == 0;
}

std::string
Deltacloud_BatchSystem::getIPAddress(const deltacloud_instance & instance) {

  bool connection_ok = false;

  // Test with previously saved address
  if (addresses.find(instance.id) != addresses.end() ) {
    std::stringstream ss;
    ss << "nc -z " << addresses[instance.id] << " 22";
    connection_ok = system(ss.str().c_str());
    if (connection_ok) {
      return addresses[instance.id];
    }
    addresses.erase(addresses.find(instance.id));
  }

  unsigned short nb_tries = 0;
  deltacloud_address * address;

  while (true) {
    // Connection to public addresses
    deltacloud_for_each(address, instance.public_addresses) {
      std::stringstream ss;
      ss << "nc -z " << address->address << " 22";
      connection_ok = system(ss.str().c_str());
      if (connection_ok) {
        addresses[instance.id] = address->address;
        return address->address;
      }
      WARNING("Connection failed on "<< address->address << "...");
    }
    if (!connection_ok) {
      // Connection to private addresses
      deltacloud_for_each(address, instance.private_addresses) {
      std::stringstream ss;
      ss << "nc -z " << address->address << " 22";
      connection_ok = system(ss.str().c_str());
        if (connection_ok) {
          addresses[instance.id] = address->address;
          return address->address;
        }
        WARNING("Connection failed on "<< address->address << "...");
      }
    }
    if (!connection_ok) {
      nb_tries++;
      if (nb_tries >= NB_TRIES_MAX) {
        return "";
      }
      sleep(10);
    }
  }
}

bool
Deltacloud_BatchSystem::runOn(const deltacloud_instance & instance,
    std::string command) {

  std::string ip = getIPAddress(instance);
  if (ip == "") {
    return false;
  }
    return runSsh(ip, command);
}

void
replaceAll(std::string& str, const std::string& from, const std::string& to) {
  if(from.empty())
    return;
  size_t start_pos = 0;
  while((start_pos = str.find(from, start_pos)) != std::string::npos) {
    str.replace(start_pos, from.length(), to);
    start_pos += to.length();
  }
}

bool
Deltacloud_BatchSystem::runDietCommand(std::vector<std::string> vm_ids,
    const char * addon_prologue, const char * command,
    diet_profile_t * profile, std::string jobid) {

  if (vm_ids.empty()) {
    ERROR_DEBUG("No VMs in runDietCommand", false);
  }


  // Put nodes list on nfs
  std::string nfs_for_this_job = getNFSPath() + jobid;
  boost::filesystem::path path_job_nfs(nfs_for_this_job);

  if (!boost::filesystem::create_directory(path_job_nfs)) {
    WARNING("Cannot create directory : " << path_job_nfs);
  }

  // Create NFS repo
  std::stringstream nfscommand;
  nfscommand << nfs_for_this_job;

  for (std::vector<std::string>::const_iterator it = vm_ids.begin(); it != vm_ids.end(); ++it) {
    deltacloud_instance instance;
    ENSURE(deltacloud_get_instance_by_id(&api, (*it).c_str(), &instance),);
    std::string ip = getIPAddress(instance);
    if (ip != "") {
      nfscommand << " " << ip << "(rw,sync,no_subtree_check)";
    }
  }
  nfscommand << " #" << jobid << std::endl;

  // Add entry to nfs export file in the SeD
  nfsfile_mutex.lock();

  std::fstream exports;
  exports.open("/etc/exports", std::fstream::out | std::fstream::app);
  if (!exports.is_open()) {
    WARNING("File /etc/exports cannot be open");
  }
  exports << nfscommand;
  exports.close();
  // Reload nfs configuration
  system("exportfs -r");
  nfsfile_mutex.unlock();

  for (std::vector<std::string>::const_iterator it = vm_ids.begin(); it != vm_ids.end(); ++it) {
    deltacloud_instance instance;
    ENSURE(deltacloud_get_instance_by_id(&api, (*it).c_str(), &instance),);
    // On VMs, the nfs directory will be the pathToNFS specified in the config
    runOn(instance, "mount $HOSTNAME:" + nfs_for_this_job + " " + pathToNFS);
  }

  // Nodes file
  boost::filesystem::path nodes_path = path_job_nfs / "nodes.txt";
  std::fstream nodes;
  nodes.open(nodes_path.c_str(), std::fstream::out);
  if (!nodes.is_open()) {
    WARNING("File " << nodes_path.c_str() << " cannot be open");
  }

  for (std::vector<std::string>::const_iterator it = vm_ids.begin(); it != vm_ids.end(); ++it) {
    deltacloud_instance instance;
    ENSURE(deltacloud_get_instance_by_id(&api, it->c_str() , &instance),);
    nodes << *it << std::endl;
  }
  nodes.close();

  // Script
  boost::filesystem::path script_path = path_job_nfs / "script.sh";

  std::fstream script;
  script.open(script_path.c_str(), std::fstream::out);
  if (!script.is_open()) {
    WARNING("File " << script_path.c_str() << " cannot be open");
  }

  script << "#!/bin/sh" << std::endl;
  if (addon_prologue != NULL) {
    script << addon_prologue << std::endl;
  }

  script << "DIET_BATCH_NODESFILE=" << nodes_path.c_str() << std::endl;
  script << "DIET_BATCH_NODESLIST=$(cat $DIET_BATCH_NODESFILE | sort | uniq)"
    << std::endl;
  script << "DIET_BATCH_NBNODES=$(echo $DIET_BATCH_NODESLIST | wc -l)"
    << std::endl;

  script << "DIET_BATCH_NBNODES=" << profile->nbprocs << std::endl;
  script << "DIET_USER_NBPROCS=" << profile->nbprocess << std::endl;
  script << "DIET_NAME_FRONTALE=" << frontalName << std::endl;
  // TODO What is the cloud job ID ?
  script << "DIET_BATCH_JOB_ID=" << batchJobID << std::endl;
  script << "DIET_BATCHNAME=" << batchName << std::endl;

  script << command << std::endl;

  script.close();

  deltacloud_instance instance;
  ENSURE(deltacloud_get_instance_by_id(&api, vm_ids[0].c_str(), &instance),);

  std::ostringstream run_command;
  run_command << "sh " << script_path;
  bool result = runOn(instance, run_command.str());

  // TODO Delete the directory path_job_nfs ?
  return result;
}

int
Deltacloud_BatchSystem::diet_submit_parallel(diet_profile_t *profile,
    const char *addon_prologue,
    const char *command){

  std::string uuid = to_string(diet_generate_uuid());


  // Reservation noeuds
  std::vector<std::string> vm_ids = instanciateVMs(profile, uuid);

  // Soumission de la commande
  runDietCommand(vm_ids, addon_prologue, command, profile, uuid);


  // Suppression des noeuds
  killVMs(vm_ids);
  cleanJob(uuid);

  return 0;
}

bool
Deltacloud_BatchSystem::cleanJob(std::string jobid) {
  // Removes nfs entry from /etc/exports
  // It has been stored as a line like : '/dir ... #<jobid>'

  nfsfile_mutex.lock();
  std::ostringstream ss;
  ss << "sed -i /#" << jobid << "$/d /etc/exports";
  bool ok = system(ss.str().c_str());
  nfsfile_mutex.unlock();
  return ok;
}


BatchSystem::batchJobState
Deltacloud_BatchSystem::askBatchJobStatus(int batchJobID){
  // TODO A implementer
  return TERMINATED;
}

int
Deltacloud_BatchSystem::isBatchJobCompleted(int batchJobID){
  // TODO A implementer
  return 0;
}

int
Deltacloud_BatchSystem::getNbTotResources(){
  // TODO Code method
  return 0;
}

int
Deltacloud_BatchSystem::getNbResources(){
  // TODO Code method
  return 0;
}

int
Deltacloud_BatchSystem::getMaxWalltime(){
  // TODO Code method
  return 0;
}

int
Deltacloud_BatchSystem::getMaxProcs(){
  // TODO Code method
  return 0;
}

int
Deltacloud_BatchSystem::getNbTotFreeResources(){
  // TODO Code method
  return 0;
}

int
Deltacloud_BatchSystem::getNbFreeResources(){
  // TODO Code method
  return 0;
}

int
Deltacloud_BatchSystem::init() {

    // Fill the credential configuration

    if (!CONFIG_STRING(diet::CLOUDURL, credentials.base_url)) {
      ERROR_DEBUG("No cloud url specified. You should add a cloudURL property\
          in the cfg file.", 1);
    }

    if (!CONFIG_STRING(diet::USERNAME, credentials.username)) {
      ERROR_DEBUG("No username specified. You should add a userName property\
          in the cfg file.", 1);
    }

    if (!CONFIG_STRING(diet::PASSWORD, credentials.password)) {
      ERROR_DEBUG("No password specified. You should add a password property\
          in the cfg file.", 1);
    }

    // Initialize the deltacloud API

    char * url = strdup(credentials.base_url.c_str());
    char * user = strdup(credentials.username.c_str());
    char * password = strdup(credentials.password.c_str());

    ENSURE(deltacloud_initialize(&api, url, user, password),);

    free(url);
    free(user);
    free(password);
  return api.initialized;
}

std::vector<std::string>
Deltacloud_BatchSystem::instanciateVMs(diet_profile_t *profile,
    std::string jobid) {

  TRACE_TEXT(TRACE_ALL_STEPS, "Image to look for : " << profile->image_name << std::endl);

  char * imageid = getImageId(profile->image_name);
  TRACE_TEXT(TRACE_ALL_STEPS, "IMAGE ID = " << imageid << std::endl);

  char * hwprofileid = getHardwareProfileId(profile->hardware_profile);
  TRACE_TEXT(TRACE_ALL_STEPS, "HARDWARE PROFILE = " << hwprofileid << std::endl);

  deltacloud_create_parameter params[1];
  ENSURE(deltacloud_prepare_parameter(&params[0], "hwp_id", hwprofileid),);

  // Instance creation
  // The first one will serve as a nfs server
  std::vector<std::string> instances;
  for (int i = 0; i < profile->nbprocs; ++i)
  {
    char * instanceid;
    ENSURE(
        deltacloud_create_instance(&api, imageid, params, 1, &instanceid),
        return instances);

    instances.push_back(instanceid);
    free(instanceid);
  }

  deltacloud_free_parameter_value(&params[0]);
  free(hwprofileid);
  free(imageid);

  // Wait for instance to run
  deltacloud_instance instance;
  bool running[instances.size()];
  memset(&running, false, sizeof(bool) * instances.size());;
  bool still_some_pending = true;

  while (still_some_pending) {
    still_some_pending = false;
    for (int i = 0; i < instances.size(); ++i) {
      const char * id = instances[i].c_str();
      TRACE_TEXT(TRACE_ALL_STEPS, "Verifying " << id << std::endl);
      if (running[i]) {
        TRACE_TEXT(TRACE_ALL_STEPS, "Running." << std::endl);
        continue;
      }
      TRACE_TEXT(TRACE_ALL_STEPS, "Pending..." << std::endl);
      still_some_pending = true;

      ENSURE(deltacloud_get_instance_by_id(&api, id, &instance),);

      if (strcmp(instance.state, "RUNNING") == 0) {
        TRACE_TEXT(TRACE_ALL_STEPS, "Instance " << id << " is now running" << std::endl);
        running[i] = true;
      }
      else {
        sleep(5);
      }
    }
  }

  // Prepare NFS
  // TODO Is it really necessary ?

  TRACE_TEXT(TRACE_ALL_STEPS, "System Ready" << std::endl);
  return instances;
}

int
Deltacloud_BatchSystem::killVMs(std::vector<std::string> vm_ids) {

  ENSURE(deltacloud_supports_instances(&api),return 1);


  for (std::vector<std::string>::iterator it = vm_ids.begin();
      it != vm_ids.end(); ++it) {
    TRACE_TEXT(TRACE_ALL_STEPS, "Killing instance " << it->c_str() << std::endl);
    deltacloud_instance instance;

    ENSURE(
        deltacloud_get_instance_by_id(&api, it->c_str(), &instance),);

    ENSURE(deltacloud_instance_stop(&api, &instance),);

    ENSURE(deltacloud_instance_destroy(&api, &instance),);
  }

  return 0;
}

char *
Deltacloud_BatchSystem::getHardwareProfileId(char * profile_name){
  struct deltacloud_hardware_profile * profiles;
  struct deltacloud_hardware_profile * profile;

  ENSURE(deltacloud_supports_hardware_profiles(&api), return NULL);

  ENSURE(deltacloud_get_hardware_profiles(&api, &profiles), return NULL);

  deltacloud_for_each(profile, profiles) {
    if (strcmp(profile->name, profile_name) == 0) {
      char * id = strdup(profile->id);
      deltacloud_free_hardware_profile_list(&profiles);
      return id;
    }
  }
  deltacloud_free_hardware_profile_list(&profiles);
  return NULL;

}

char *
Deltacloud_BatchSystem::getImageId(char * image_name){
  struct deltacloud_image * images;
  struct deltacloud_image * image;

  ENSURE(deltacloud_supports_images(&api), return NULL);

  ENSURE(deltacloud_get_images(&api, &images), return NULL);

  deltacloud_for_each(image, images) {
    if (strcmp(image->name, image_name) == 0) {
      char * id = strdup(image->id);
      deltacloud_free_image_list(&images);
      return id;
    }
  }
  deltacloud_free_image_list(&images);
  return NULL;

}
