/**
* @file configuration.cc
* 
* @brief  Configuration manager 
* 
* @author  Haikel Guemar (haikel.guemar@sysfera.com)
* 
* @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/



#include <boost/regex.h>
#include <stdexcept>

#include "configuration.hh"
#include "constants.hh"

const std::string simple_cast_traits<std::string>::zero_value = "";
/**
 * @param[in]  param
 * @param[out] value result
 * @return param has been set or not
 */
template<>
bool
getConfigValue(diet::param_type_t param, std::string& value)
{
  const std::string& key = (diet::params)[param].value;
  ConfigMap::iterator it = GetConfigPtr()->find(key);
  if (GetConfigPtr()->end() == it) {
    return false;
  } else {
    value = it->second;
    return true;
  }
}

/**
 * @param[in]  param
 * @param[out] value result
 * @return param has been set or not
 * TODO: not handled by generic method
 */
bool
getAddressConfigValue(diet::param_type_t param, std::string& value) {
  const std::string& key = (diet::params)[param].value;
  ConfigMap::iterator it = GetConfigPtr()->find(key);
  if (GetConfigPtr()->end() == it) {
    return false;
  } else {
    std::string val = it->second;
    regex_t *preg = new regex_t;
    // check that our address is properly formatted host:port
    // if not return an empty string
    regcomp(preg, "^[-_.a-zA_Z0-9]+?:[0-9]+$", REG_EXTENDED | REG_NOSUB);
    int res = regexec(preg, val.c_str(), 0, 0, 0);
    if (res != 0) {
      return false;
    }
    delete preg;

    value = val;
    return true;
  }
}


/**
 * @param[in]  param
 * @param[out] value result
 * @return param has been set or not
 * TODO: not handled by generic method
 */
bool
getAgentConfigValue(diet::param_type_t param, std::string& value) {
  const std::string& key = (diet::params)[param].value;
  ConfigMap::iterator it = GetConfigPtr()->find(key);
  if (GetConfigPtr()->end() == it) {
    return false;
  } else {
    std::string& val = it->second;
    if ((val == "LA") ||
        (val == "DIET_LOCAL_AGENT") ||
        (val == "MA") ||
        (val == "DIET_MASTER_AGENT") ||
        (val == "MA_DAG") ||
        (val == "DIET_MA_DAG")) {
      value = val;
    } else {
      // FIXME: unknown agent type
      // actually throw an exception but might not be an appropriate behavior
      throw std::runtime_error("Unknown agent type, known types are "
                               "DIET_LOCAL_AGENT (LA), DIET_MASTER_AGENT (MA)"
                               "DIET_MA_DAG (MA_DAG)");
    }
  }
  return true;
}
