/**
 * @file AdvancedDagdaComponent.cc
 *
 * @brief  Advanced Dagda component implementation
 *
 * @author  Gael Le Mahec (lemahec@clermont.in2p3.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

#include "AdvancedDagdaComponent.hh"
#include "DIET_data_internal.hh"

#include <iostream>
#include <fstream>
#include <ctime>
#include "debug.hh"
// #include "ORBMgr2.hh"


using namespace std;

// modif bisnard_logs_1
char *
AdvancedDagdaComponent::sendFile(const corba_data_t &data,
                                 const char *destName) {
  struct timeval tv1;
  struct timeval tv2;
  char *ret;

  gettimeofday(&tv1, NULL);
  ret = DagdaImpl::sendFile(data, destName);
  gettimeofday(&tv2, NULL);

  double elapsed = (tv2.tv_sec - tv1.tv_sec) * 1000000
                   + (tv2.tv_usec - tv1.tv_usec);
  TRACE_TEXT(
    TRACE_ALL_STEPS, "Data " << data.desc.id.idNumber
                             << " transfered in " << elapsed <<
    " microsecond(s)." << endl);

  // if (getLogComponent())
  // getLogComponent()->logDataTransferTime(data.desc.id.idNumber,
  // dest->getID(),
  // (unsigned long) elapsed);


  if (stats != NULL && elapsed != 0) {
    stats->addStat(getIDstr(), destName,
                   ((double) data_sizeof(&data.desc)) / elapsed);
    stats->addStat(string(destName), getIDstr(),
                   ((double) data_sizeof(&data.desc)) / elapsed);
  }
  return ret;
} // sendFile

char *
AdvancedDagdaComponent::sendData(const char *ID, const char *destName) {
  struct timeval tv1;
  struct timeval tv2;
  char *ret;
  corba_data_t *data = getData(ID);
  size_t dataSize = data_sizeof(&data->desc);

  gettimeofday(&tv1, NULL);
  ret = DagdaImpl::sendData(ID, destName);
  gettimeofday(&tv2, NULL);

  double elapsed = (tv2.tv_sec - tv1.tv_sec) * 1000000
                   + (tv2.tv_usec - tv1.tv_usec);
  TRACE_TEXT(
    TRACE_ALL_STEPS, "Data " << data->desc.id.idNumber
                             << " transfered in " << elapsed <<
    " microsecond(s)." << endl);

  // if (getLogComponent())
  // getLogComponent()->logDataTransferTime(data->desc.id.idNumber,
  // dest->getID(),
  // (unsigned long) elapsed);

  if (stats != NULL && elapsed != 0) {
    stats->addStat(getIDstr(), destName,
                   ((double) dataSize) / elapsed);
    stats->addStat(destName, getIDstr(),
                   ((double) dataSize) / elapsed);
  }
  return ret;
} // sendData
// end modif bisnard_logs_1

void
AdvancedDagdaComponent::lclAddData(const char *srcName,
                                   const corba_data_t &data) {
  try {
    SimpleDagdaImpl::lclAddData(srcName, data);
    setRegisterTime(data.desc.id.idNumber);
    if (shareFiles && DGD_OBJ_TYPE(data) == DIET_FILE_OBJ) {
      shareData(data);
    }
  } catch (Dagda::NotEnoughSpace &ex) {
    TRACE_TEXT(
      TRACE_ALL_STEPS, "Needs more space. Try to call the selected cache "
      << "algorithm." << endl);
    if (mngFunction != NULL) {
      size_t needed = data_sizeof(&data.desc);
      size_t max;
      size_t used;
      if (DGD_OBJ_TYPE(data) == DIET_MEM_OBJ) {
        max = getMemMaxSpace();
        used = getUsedMemSpace();
      } else {
        max = getDiskMaxSpace();
        used = getUsedDiskSpace();
      }
      if (used + needed > max) {
        needed += used - max;
      }

      if (mngFunction(this, needed, DGD_OBJ_TYPE(data))) {
        throw Dagda::NotEnoughSpace(ex.available);
      }
      SimpleDagdaImpl::lclAddData(srcName, data);
      setRegisterTime(data.desc.id.idNumber);
      if (shareFiles && DGD_OBJ_TYPE(data) == DIET_FILE_OBJ) {
        shareData(data);
      }
    } else {
      throw Dagda::NotEnoughSpace(ex.available);
    }
  }
} // lclAddData

void
AdvancedDagdaComponent::lclRemData(const char *dataID){

	SimpleDagdaImpl::lclRemData(dataID);

	registerMutex.lock();
	registerTime.erase(dataID);
	registerMutex.unlock();

	lastUsageMutex.lock();
	lastUsageTime.erase(dataID);
	lastUsageMutex.unlock();

	nbUsageMutex.lock();
	nbUsage.erase(dataID);
	nbUsageMutex.unlock();

}



void
AdvancedDagdaComponent::registerFile(const corba_data_t &data) {
  if (strcmp(data.desc.specific.file().path, "") == 0) {
    throw Dagda::InvalidPathName(data.desc.id.idNumber,
                                 data.desc.specific.file().path);
  }
  ifstream file(data.desc.specific.file().path);
  if (!file.is_open()) {
    throw Dagda::UnreachableFile(data.desc.specific.file().path);
  }
  file.close();
  addData(data);
  unlockData(data.desc.id.idNumber);
} // registerFile

corba_data_t *
AdvancedDagdaComponent::getData(const char *dataID) {
  string id(dataID);

  setUsageTime(id);
  incNbUsage(dataID);

  return SimpleDagdaImpl::getData(dataID);
}

char *
AdvancedDagdaComponent::getBestSource(const char *dest, const char *dataID) {
  SeqString *managers = pfmGetDataManagers(dataID);

  if (managers->length() == 0) {
    throw Dagda::DataNotFound(dataID);
  }

  if (stats != NULL) {
    TRACE_TEXT(TRACE_ALL_STEPS, "Data " << dataID << " has " <<
               managers->length() << " replica(s) on the platform." << endl);
    double maxStat = 0;
    string found = string((*managers)[0]);

    for (unsigned int i = 0; i < managers->length(); ++i) {
      string curMngr = string((*managers)[i]);
      double curStat = stats->getStat(string(getID()), curMngr);
      if (stats->cmpStats(curStat, maxStat)) {
        maxStat = curStat;
        found = curMngr;
      }
    }
    return CORBA::string_dup(found.c_str());
  }

  return SimpleDagdaImpl::getBestSource(dest, dataID);
} // getBestSource

clock_t
AdvancedDagdaComponent::getRegisterTime(const char *dataID) {
  registerMutex.lock();
  clock_t ret = clock();
  if (registerTime.find(dataID) != registerTime.end()) {
    ret = registerTime[dataID];
  }
  registerMutex.unlock();
  return ret;
}

void
AdvancedDagdaComponent::setRegisterTime(const char *dataID, clock_t time) {
  registerMutex.lock();
  registerTime[dataID] = time;
  registerMutex.unlock();
}

void
AdvancedDagdaComponent::setRegisterTime(const char *dataID) {
  setRegisterTime(dataID, clock());
}

clock_t
AdvancedDagdaComponent::getLastUsageTime(const char *dataID) {
  lastUsageMutex.lock();
  clock_t ret = clock();
  if (lastUsageTime.find(dataID) != lastUsageTime.end()) {
    ret = lastUsageTime[dataID];
  }
  lastUsageMutex.unlock();
  return ret;
}

void
AdvancedDagdaComponent::setUsageTime(string id, clock_t time) {
  lastUsageMutex.lock();
  lastUsageTime[id] = time;
  lastUsageMutex.unlock();
}

void
AdvancedDagdaComponent::setUsageTime(string id) {
  setUsageTime(id, clock());
}

unsigned long
AdvancedDagdaComponent::getNbUsage(const char *dataID) {
  nbUsageMutex.lock();
  unsigned long ret = 0;
  if (nbUsage.find(dataID) != nbUsage.end()) {
    ret = nbUsage[dataID];
  }
  nbUsageMutex.unlock();
  return ret;
}

void
AdvancedDagdaComponent::incNbUsage(const char *dataID) {
  nbUsageMutex.lock();
  nbUsage[dataID]++;
  nbUsageMutex.unlock();
}

void
AdvancedDagdaComponent::shareData(const corba_data_t &data) {
  std::map<string, Dagda_ptr>::iterator itch;

  childrenMutex.lock();
  for (itch = getChildren()->begin(); itch != getChildren()->end();)
    try {
      (*itch).second->registerFile(data);
      ++itch;
    } catch (CORBA::COMM_FAILURE &e1) {
      getChildren()->erase(itch++);
    } catch (CORBA::TRANSIENT &e2) {
      getChildren()->erase(itch++);
    } catch (Dagda::InvalidPathName &e3) {
      WARNING(
        "Fail to share the file \"" << e3.path <<
        "\" (invalid path name) on " <<
        (*itch).second->getID() << " data manager.");
    } catch (Dagda::UnreachableFile &e4) {
      WARNING(
        "File \"" << e4.path << "\" is unreachable on " <<
        (*itch).second->getID() <<
        " data manager.");
    }
  childrenMutex.unlock();
} // shareData
