/**
 * @file dietFwdr.cc
 *
 * @brief   DIET forwarder implementation - Forwarder executable
 *
 * @author  Gael Le Mahec (gael.le.mahec@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

#include "DIETForwarder.hh"
#include "ORBMgr.hh"
#include "configuration.hh"
#include "DIET_grpc.h"
#include "SSHTunnel.hh"
#include "Options.hh"

#include "dietFwdr.hh"

#include "debug.hh"

#include <iostream>
#include <cstdlib>
#include <string>
#include <cstring>
#include <sstream>
#include <fstream>
#include <algorithm>

#include <omniORB4/CORBA.h>

#include "OSIndependance.hh"     // For sleep function
#include "DIET_uuid.hh"

#ifdef __WIN32__
#define sleep(value) (Sleep(value*1000))
#endif

template <typename C>
class CStringInserter {
private:
C &c_;
public:
explicit
CStringInserter(C &c): c_(c) {
}

void
operator()(const char *cstr) {
  c_.push_back(strdup(cstr));
}

void
operator()(std::ostringstream &oss) {
  char *cstr = strdup(oss.str().c_str());
  c_.push_back(cstr);
}
};


int processConfig(FwrdConfig *cfg) {

	/* Process the network config file if relevant before doing
	 * anything with the options */
	const std::string &configFile = cfg->getCfgPath();
	if (!configFile.empty()) {
		FileParser fileParser;
		try {
			fileParser.parseFile(configFile);
		} catch (...) {
			ERROR_DEBUG(
					"Missing file or parsing error for network config file: " << configFile,
					GRPC_CONFIGFILE_ERROR);
		}

		CONFIGMAP = fileParser.getConfiguration();

		std::string opt_name, opt_peerName, opt_sshHost, opt_remoteHost,
				opt_remotePort, opt_sshPort, opt_sshLogin, opt_sshKey,
				opt_peerIOR;

		int opt_tunnelWait;
		unsigned long opt_nbRetry;

		/* Not adding them to the "internal structure" because the options are only for the forwarder
		 * and shouldn't have an impact on the ORB stuff */
		if (CONFIG_STRING(diet::NAME, opt_name)) {
			cfg->setName(opt_name);
		}

		if (CONFIG_STRING(diet::PEERNAME, opt_peerName)) {
			cfg->setPeerName(opt_peerName);
		}

		if (CONFIG_STRING(diet::SSHHOST, opt_sshHost)) {
			cfg->setSshHost(opt_sshHost);
		}

		if (CONFIG_STRING(diet::REMOTEHOST, opt_remoteHost)) {
			cfg->setRemoteHost(opt_remoteHost);
		}

		if (CONFIG_STRING(diet::REMOTEPORT, opt_remotePort)) {
			cfg->setRemotePortFrom(opt_remotePort);
		}

		if (CONFIG_INT(diet::TUNNELWAIT, opt_tunnelWait)) {
			if (opt_tunnelWait < 0) {
				cfg->setWaitingTime(0);
			} else {
				cfg->setWaitingTime(opt_tunnelWait);
			}
		}

		if (CONFIG_STRING(diet::SSHPORT, opt_sshPort)) {
			cfg->setSshPort(opt_sshPort);
		}

		if (CONFIG_STRING(diet::SSHLOGIN, opt_sshLogin)) {
			cfg->setSshLogin(opt_sshLogin);
		}

		if (CONFIG_STRING(diet::SSHKEY, opt_sshKey)) {
			cfg->setSshKeyPath(opt_sshKey);
		}

		if (CONFIG_ULONG(diet::NBRETRY, opt_nbRetry)) {
			cfg->setNbRetry(opt_nbRetry);
		}

		if (CONFIG_STRING(diet::PEERIOR, opt_peerIOR)) {
			cfg->setPeerIOR(opt_peerIOR);
		}

	}

	if (!cfg->getSshHost().empty()) {
		cfg->createTo(true);
		cfg->createFrom(true);
	}

	if (cfg->getName().empty()) {
		std::ostringstream name;
		char host[256];
		boost::uuids::uuid uuid = diet_generate_uuid();

		gethostname(host, 256);
		host[255] = '\0';

		std::transform(host, host + strlen(host), host, change);
		name << "Forwarder-" << host << "-" << uuid;
		WARNING(
				"Missing parameter: name (use --name to fix it)\n" << "Use default name: " << name.str() << "\"n");
		cfg->setName(name.str());
	}

	if (cfg->createFrom()) {
		if (cfg->getPeerName().empty() || cfg->getSshHost().empty()) {
			ERROR_DEBUG(
					"Missing parameter(s) to create tunnel." << " Mandatory parameters:\n" << '\t' << "- Peer name (--peer-name <name>)\n" << '\t' << "- SSH host (--ssh-host <host>)",
					EXIT_FAILURE);
		}
	}

	return EXIT_SUCCESS;
}


int
main(int argc, char *argv[], char *envp[]) {

  std::vector<char *> args;
  CStringInserter<std::vector<char *> > ins(args);

  /* Forwarder configuration. */
  FwrdConfig cfg(argv[0]);

  Options opt(&cfg, argc, argv, envp);
  /* Mandatory parameter. */
  opt.setOptCallback("--name", name);

  /* Mandatory when creating tunnels. */
  opt.setOptCallback("--peer-name", peer_name);
  opt.setOptCallback("--ssh-host", ssh_host);
  /* Optional, set to "localhost" by default. */
  opt.setOptCallback("--remote-host", remote_host);
  /* Optional, we try to determine it automatically. */
  opt.setOptCallback("--remote-port", remote_port_from);
  /* Optional, set waiting time for tunnel creation */
  opt.setOptCallback("--tunnel-wait", tunnel_wait);

  /* Optional config file, this one is called net-config to avoid breaking the interface with
   * GoDiet, however it only contains the same stuff as the command line options */
  opt.setOptCallback("--net-config", cfg_path);
  opt.setOptCallback("--config", cfg_path);
  /* Optional config file, that contains omniORB config parameters
   * Here, it is only used to retrieve the DIETHOSTNAME and the DIETPORT */
  opt.setOptCallback("--orb-config", orb_cfg_path);


  /* Optional */
  opt.setOptCallback("--ssh-port", ssh_port);
  opt.setOptCallback("--ssh-login", ssh_login);
  opt.setOptCallback("--ssh-key", key_path);

  /* Optional parameters/flags. */
  opt.setOptCallback("--nb-retry", nb_retry);
  opt.setOptCallback("--peer-ior", peer_ior);


  opt.processOptions();

  int returnValue;
  returnValue = processConfig(&cfg);
  if (returnValue != EXIT_SUCCESS){
	  return returnValue;
  }


  SSHTunnel tunnel;
  DIETForwarder *forwarder;
  try {
    forwarder = new DIETForwarder(cfg.getName());
  } catch (std::exception &e) {
    ERROR_DEBUG(e.what(), EXIT_FAILURE);
  }

  // get ORB configuration file
  const std::string &configFile = cfg.getOrbCfgPath();

  FileParser fileParser;
  if (!configFile.empty()){
	  try {
	  fileParser.parseFile(configFile);
	  } catch (...) {
		  ERROR_DEBUG("Missing file or error parsing error ORB config file: " << configFile, GRPC_CONFIGFILE_ERROR);
	  }
  }
  CONFIGMAP = fileParser.getConfiguration();

  /* Copy input parameters into internal structure */
  for (int i = 0; i < argc; i++) {
    ins(argv[i]);
  }

  /* Get listening port & hostname */
  int port;
  std::string host;
  bool hasPort = CONFIG_INT(diet::DIETPORT, port);
  bool hasHost = CONFIG_STRING(diet::DIETHOSTNAME, host);
  if (hasPort || hasHost) {
      std::ostringstream endpoint;
      ins("-ORBendPoint");
      endpoint << "giop:tcp:" << host << ":";
      if (hasPort) {
        endpoint << port;
      }
      ins(endpoint);
  } else {
    ins("-ORBendPointPublish");
    ins("all(addr)");
  }


  ORBMgr::init(args.size(), &args[0]);
  std::string ior;
  int count = 0;

  ORBMgr::get().activate(forwarder);
  do {
    try {
      ORBMgr::get().bind(FWRDCTXT, cfg.getName(), forwarder->_this(), true);
      break;
    } catch (CORBA::TRANSIENT &err) {
      TRACE_TEXT(TRACE_MAIN_STEPS,
                 "Error when binding the forwarder "
                 << cfg.getName() << std::endl);
      if (count++ < cfg.getNbRetry()) {
        sleep(5);
        continue;
      }
      ORBMgr::get().deactivate(forwarder);
      return EXIT_FAILURE;
    }
  } while (true);

  /* Write the IOR to a file on /tmp. */
  ior = ORBMgr::get().getIOR(forwarder->_this());
  std::string iorFilename("/tmp/DIET-forwarder-ior-");
  iorFilename += cfg.getName() + ".tmp";
  std::ofstream of(iorFilename.c_str(), std::ios_base::trunc);

  if (!of.is_open()) {
    WARNING("cannot open file " << iorFilename
                                << " to store the IOR");
  } else {
    TRACE_TEXT(TRACE_MAIN_STEPS, "Write IOR to " << iorFilename << std::endl);
    if (cfg.createFrom()) {  // Creating tunnel(s)
      std::istringstream is(cfg.getRemotePortFrom());
      int port;

      is >> port;
      of << ORBMgr::convertIOR(ior, cfg.getRemoteHost(), port);
    } else {  // Waiting for connexion.
      of << ior;
    }
    of << std::endl;
    of << freeTCPport();  // also write a free port
    of.close();
  }
  TRACE_TEXT(TRACE_MAIN_STEPS, "Forwarder: " << ior << std::endl);

  tunnel.setSshHost(cfg.getSshHost());
  tunnel.setRemoteHost(cfg.getRemoteHost());

  tunnel.setSshPath(cfg.getSshPath());
  tunnel.setSshPort(cfg.getSshPort());
  tunnel.setSshLogin(cfg.getSshLogin());
  tunnel.setSshKeyPath(cfg.getSshKeyPath());

  tunnel.setWaitingTime(cfg.getWaitingTime());

  /* Manage the peer IOR. */
  if (cfg.getPeerIOR().empty() && cfg.createFrom()) {
    /* Try to retrieve the peer IOR. */
    SSHCopy copy(cfg.getSshHost(),
                 "/tmp/DIET-forwarder-ior-" + cfg.getPeerName() + ".tmp",
                 "/tmp/DIET-forwarder-ior-" + cfg.getPeerName() + ".tmp");
    copy.setSshPath("/usr/bin/scp");
    copy.setSshPort(cfg.getSshPort());
    copy.setSshLogin(cfg.getSshLogin());
    copy.setSshKeyPath(cfg.getSshKeyPath());
    try {
      if (copy.getFile()) {
        TRACE_TEXT(TRACE_MAIN_STEPS, "Got remote IOR file" << std::endl);
        cfg.setPeerIOR("/tmp/DIET-forwarder-ior-" + cfg.getPeerName() + ".tmp");
      } else {
        ERROR_DEBUG("Could not get remote IOR file.\n"
              << "Please check that you can scp files"
              << "between the ssh host and this host, _n"
              << "or specify the remote IOR with the following option:\n"
              << "\t- Remote IOR (--peer-ior <IOR>)", EXIT_FAILURE);
      }
    } catch (...) {
      TRACE_TEXT(TRACE_MAIN_STEPS,
                 "Got an exception while retrieving IOR file" << std::endl);
    }
  }
  if (!cfg.getPeerIOR().empty() && cfg.getPeerIOR().find("IOR:") != 0) {
    /* Extract the IOR from a file. */
    std::ifstream file(cfg.getPeerIOR().c_str());
    std::string peerIOR;
    std::string peerPort;
    if (!file.is_open()) {
      ERROR_DEBUG("Error: Invalid peer-ior parameter", EXIT_FAILURE);
    }
    file >> peerIOR;
    cfg.setPeerIOR(peerIOR);
    if (!file.eof() && (cfg.getRemotePortFrom().empty())) {
      file >> peerPort;
      cfg.setRemotePortFrom(peerPort);
    }
  }

  if (!cfg.getPeerIOR().empty()) {
    tunnel.setRemotePortTo(ORBMgr::getPort(cfg.getPeerIOR()));
  } else {
    tunnel.setRemotePortTo(cfg.getRemotePortTo());
  }
  if (cfg.getRemoteHost() == "auto") {
    if (!cfg.getPeerIOR().empty()) {
      tunnel.setRemoteHost(ORBMgr::getHost(cfg.getPeerIOR()));
    } else {
			tunnel.setRemoteHost(HOST_127_0_0_1);
    }
  } else {
    if (!cfg.getRemoteHost().empty()) {
      tunnel.setRemoteHost(cfg.getRemoteHost());
    } else {
			tunnel.setRemoteHost(LOCALHOST);
    }
  }

  tunnel.setRemotePortFrom(cfg.getRemotePortFrom());
  // tunnel.setLocalPortFrom(cfg.getLocalPortFrom());
  if (cfg.createFrom()) {
    if (cfg.getRemotePortFrom().empty()) {
      ERROR_DEBUG("Failed to automatically determine a remote free port.\n"
            << " You need to specify the remote port:\n"
            << '\t' << "- Remote port (--remote-port <port>)", EXIT_FAILURE);
    }
  }


  tunnel.setLocalPortTo(ORBMgr::getPort(ior));

  if (!cfg.getSshHost().empty()) {
    tunnel.createTunnelTo(cfg.createTo());
    tunnel.createTunnelFrom(cfg.createFrom());
  }
  tunnel.open();

  /* Try to find the peer. */
  bool canLaunch = true;
  if (!cfg.getPeerIOR().empty()) {
    try {
      if (connectPeer(ior, cfg.getPeerIOR(),
                      LOCALHOST,
					tunnel.getRemoteHost(),
                      tunnel.getLocalPortFrom(), tunnel.getRemotePortFrom(),
                      forwarder, &ORBMgr::get())) {
        /* In this case it seems that there is a problem with
         * the alias 'localhost', thus try to use 127.0.0.1
         */
				if (tunnel.getRemoteHost() == LOCALHOST) {
					tunnel.setRemoteHost(HOST_127_0_0_1);
        }
        if (connectPeer(ior, cfg.getPeerIOR(),
                        HOST_127_0_0_1,
						tunnel.getRemoteHost(),
                        tunnel.getLocalPortFrom(), tunnel.getRemotePortFrom(),
                        forwarder, &ORBMgr::get())) {
          TRACE_TEXT(TRACE_MAIN_STEPS, "Unable to contact remote peer."
                     << "Waiting for connection...\n");
        }
      }
    } catch (...) {
      TRACE_TEXT(TRACE_MAIN_STEPS,
                 "Error while connecting to remote peer\n");
      canLaunch = false;
    }
  }

  if (canLaunch) {
    try {
      ORBMgr::get().wait();
      ORBMgr::get().unbind(FWRDCTXT, cfg.getName());
    } catch (...) {
      WARNING("Error while exiting the ORBMgr::wait() function");
    }
  }

  /**
   * DO NOT KILL MANUALLY !!!
   * singleton will take care of cleanup automatically,
   * manually calling kill can cause issue (ORBMgr is released
   * before AgentImpl, causing ~AgentImpl to call null object)
   */
  // ORBMgr::kill();

  TRACE_TEXT(TRACE_MAIN_STEPS, "Forwarder is now terminated" << std::endl);

  return EXIT_SUCCESS;
} // main

int
connectPeer(const std::string &ior, const std::string &peerIOR,
            const std::string &newHost, const std::string &remoteHost,
            int localPortFrom, int remotePortFrom,
            DIETForwarder *forwarder, ORBMgr *mgr) {
  std::string newPeerIOR = ORBMgr::convertIOR(peerIOR, newHost, localPortFrom);

  Forwarder_var peer;

  peer = mgr->resolve<Forwarder, Forwarder_var>(newPeerIOR);

  try {
    peer->connectPeer(ior.c_str(), remoteHost.c_str(), remotePortFrom);
    forwarder->setPeer(peer);

    // Get the existing contexts except the Forwarders one
    std::list<std::string> contexts = mgr->contextList();
    contexts.remove(FWRDCTXT);
    // Get the other forwarder list
    std::list<std::string> fwds = forwarder->otherForwarders();
    std::string fwdName = forwarder->getName();
    std::string fwdTag = "@" + fwdName;

    // For each context
    for (std::list<std::string>::const_iterator it = contexts.begin();
         it != contexts.end(); ++it) {
      // Get the local objects
      std::list<std::string> objects = mgr->localObjects(*it);
      // Get the object from other forwarders
      for (std::list<std::string>::const_iterator jt = fwds.begin();
           jt != fwds.end(); ++jt) {
        std::list<std::string> fwdObjects = mgr->forwarderObjects(*jt, *it);
        objects.insert(objects.end(), fwdObjects.begin(), fwdObjects.end());
      }

      // Bind them on the peer
      for (std::list<std::string>::const_iterator jt = objects.begin();
           jt != objects.end(); ++jt) {
        std::string objName = *it + "/" + *jt;
        std::string ior = mgr->getIOR(*it, *jt);
        forwarder->bind(objName.c_str(), ior.c_str());
      }
      // Then, get the objects binded on the peer
      SeqString *remoteObjs = peer->getBindings(it->c_str());
      // And bind them locally and on others forwarders if they are not yet
      for (unsigned int i = 0; i < remoteObjs->length(); i += 2) {
        std::string name((*remoteObjs)[i]);
        std::string ior((*remoteObjs)[i + 1]);
        if (find(objects.begin(), objects.end(), name) != objects.end()) {
          continue;
        }
        std::string newIOR = ORBMgr::convertIOR(ior, fwdTag, 0);
        mgr->bind(*it, name, newIOR, true);
        mgr->fwdsBind(*it, name, newIOR, fwdName);
      }
    }
  } catch (CORBA::TRANSIENT &err) {
    ERROR_DEBUG(
      "Unable to contact remote peer using '" << newHost <<
      "' as a \"new remote host\"", 1);
  }

  TRACE_TEXT(TRACE_MAIN_STEPS,
             "Contacted remote peer using '"
             << newHost << "' as new remote host\n");
  return 0;
} // connectPeer


void name(const std::string &name, Configuration *cfg) {
	static_cast<FwrdConfig *>(cfg)->setName(name);
}

void
peer_name(const std::string &name, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setPeerName(name);
}

void
peer_ior(const std::string &ior, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setPeerIOR(ior);
}

void
ssh_host(const std::string &host, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setSshHost(host);
}

void
remote_host(const std::string &host, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setRemoteHost(host);
}

void
remote_port_to(const std::string &port, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setRemotePortTo(port);
}

void
remote_port_from(const std::string &port, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setRemotePortFrom(port);
}

void
local_port_from(const std::string &port, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setLocalPortFrom(port);
}

void
ssh_path(const std::string &path, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setSshPath(path);
}

void
ssh_port(const std::string &port, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setSshPort(port);
}

void
ssh_login(const std::string &login, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setSshLogin(login);
}

void
key_path(const std::string &path, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setSshKeyPath(path);
}

void
tunnel_wait(const std::string &time, Configuration *cfg) {
  std::istringstream is(time);
  int n;
  is >> n;
  static_cast<FwrdConfig *>(cfg)->setWaitingTime(n);
}

void
cfg_path(const std::string &path, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setCfgPath(path);
}

void
orb_cfg_path(const std::string &path, Configuration *cfg) {
  static_cast<FwrdConfig *>(cfg)->setOrbCfgPath(path);
}

void
nb_retry(const std::string &nb, Configuration *cfg) {
  std::istringstream is(nb);
  int n;
  is >> n;
  static_cast<FwrdConfig *>(cfg)->setNbRetry(n);
}

/* Fwdr configuration implementation. */
FwrdConfig::FwrdConfig(const std::string &pgName): Configuration(pgName) {
  createTunnelTo = false;
  createTunnelFrom = false;
  nbRetry = 3;
  waitingTime = 0;
}

const std::string &
FwrdConfig::getName() const {
  return name;
}
const std::string &
FwrdConfig::getPeerName() const {
  return peerName;
}

const std::string &
FwrdConfig::getPeerIOR() const {
  return peerIOR;
}

const std::string &
FwrdConfig::getSshHost() const {
  return sshHost;
}

const std::string &
FwrdConfig::getRemoteHost() const {
  return remoteHost;
}

const std::string &
FwrdConfig::getRemotePortTo() const {
  return remotePortTo;
}

const std::string &
FwrdConfig::getRemotePortFrom() const {
  return remotePortFrom;
}

const std::string &
FwrdConfig::getLocalPortFrom() const {
  return localPortFrom;
}

bool
FwrdConfig::createTo() const {
  return createTunnelTo;
}

bool
FwrdConfig::createFrom() const {
  return createTunnelFrom;
}

const std::string &
FwrdConfig::getSshPath() const {
  return sshPath;
}

const std::string &
FwrdConfig::getSshPort() const {
  return sshPort;
}
const std::string &
FwrdConfig::getSshLogin() const {
  return sshLogin;
}

const std::string &
FwrdConfig::getSshKeyPath() const {
  return sshKeyPath;
}

int
FwrdConfig::getNbRetry() const {
  return nbRetry;
}

unsigned int
FwrdConfig::getWaitingTime() const {
  return waitingTime;
}

const std::string &
FwrdConfig::getCfgPath() const {
  return cfgPath;
}

const std::string &
FwrdConfig::getOrbCfgPath() const {
  return orbCfgPath;
}


void
FwrdConfig::setName(const std::string &name) {
  this->name = name;
}

void
FwrdConfig::setPeerName(const std::string &name) {
  this->peerName = name;
}

void
FwrdConfig::setPeerIOR(const std::string &ior) {
  this->peerIOR = ior;
}

void
FwrdConfig::setSshHost(const std::string &host) {
  this->sshHost = host;
}

void
FwrdConfig::setRemoteHost(const std::string &host) {
  this->remoteHost = host;
}

void
FwrdConfig::setRemotePortTo(const std::string &port) {
  this->remotePortTo = port;
}

void
FwrdConfig::setRemotePortFrom(const std::string &port) {
  this->remotePortFrom = port;
}

void
FwrdConfig::setLocalPortFrom(const std::string &port) {
  this->localPortFrom = port;
}

void
FwrdConfig::createTo(bool create) {
  this->createTunnelTo = create;
}

void
FwrdConfig::createFrom(bool create) {
  this->createTunnelFrom = create;
}

void
FwrdConfig::setSshPath(const std::string &path) {
  this->sshPath = path;
}

void
FwrdConfig::setSshPort(const std::string &port) {
  this->sshPort = port;
}

void
FwrdConfig::setSshLogin(const std::string &login) {
  this->sshLogin = login;
}

void
FwrdConfig::setSshKeyPath(const std::string &path) {
  this->sshKeyPath = path;
}

void
FwrdConfig::setNbRetry(const int nb) {
  this->nbRetry = nb;
}

void
FwrdConfig::setWaitingTime(const unsigned int time) {
  this->waitingTime = time;
}

void
FwrdConfig::setCfgPath(const std::string &path) {
  this->cfgPath = path;
}

void
FwrdConfig::setOrbCfgPath(const std::string &path) {
  this->orbCfgPath = path;
}


int
change(int c) {
  if (c == '.') {
    return '-';
  }
  return c;
}
