/**
* @file  WfDataWriter.hh
* 
* @brief  Abstract interface and implementations of data writers 
* 
* @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr)
* 
* @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/


#ifndef _WFDATAWRITER_HH_
#define _WFDATAWRITER_HH_

#include <string>
#include "WfUtils.hh"

/*****************************************************************************/
/*                    WfDataWriter class (ABSTRACT)                          */
/*****************************************************************************/

class WfDataWriter {
public:
  explicit WfDataWriter(std::ostream& output);

  virtual ~WfDataWriter();

  virtual void
  startContainer() = 0;

  virtual void
  endContainer() = 0;

  virtual void
  itemValue(const std::string& valueStr) = 0;

  virtual void
  itemValue(void * valuePtr,
            WfCst::WfDataType valueType = WfCst::TYPE_UNKNOWN) = 0;

  virtual void
  matrix(void * valuePtr, WfCst::WfDataType valueType,
         unsigned int nbRows, unsigned int nbColumns,
         diet_matrix_order_t order) = 0;

  virtual void
  vector(void * valuePtr, WfCst::WfDataType valueType, unsigned int size) = 0;

  virtual void
  voidElement() = 0;

  virtual void
  error() = 0;

protected:
  void
  rawValue(const std::string& valueStr);

  void
  rawValue(void * valuePtr, WfCst::WfDataType valueType);

  std::ostream& myOutput;
};

class WfXMLDataWriter : public WfDataWriter  {
public:
  explicit WfXMLDataWriter(std::ostream& output);

  virtual void
  startContainer();

  virtual void
  endContainer();

  virtual void
  itemValue(const std::string& valueStr);

  virtual void
  itemValue(void * valuePtr, WfCst::WfDataType valueType = WfCst::TYPE_UNKNOWN);

  virtual void
  matrix(void * valuePtr, WfCst::WfDataType valueType,
         unsigned int nbRows, unsigned int nbColumns,
         diet_matrix_order_t order);

  virtual void
    vector(void * valuePtr, WfCst::WfDataType valueType, unsigned int size);


  virtual void
  voidElement();

  virtual void
  error();

  // additional methods to include dataId attribute in tags
  void
  startContainer(const std::string& dataID);

  void
  itemValue(const std::string& valueStr, const std::string& dataID);

  void
  itemValue(void * valuePtr, WfCst::WfDataType valueType,
            const std::string& dataID);
};

class WfListDataWriter : public WfDataWriter  {
public:
  explicit WfListDataWriter(std::ostream& output);

  virtual void
  startContainer();

  virtual void
  endContainer();

  virtual void
  itemValue(const std::string& valueStr);

  virtual void
  itemValue(void * valuePtr, WfCst::WfDataType valueType = WfCst::TYPE_UNKNOWN);

  virtual void
  matrix(void * valuePtr, WfCst::WfDataType valueType,
         unsigned int nbRows, unsigned int nbColumns,
         diet_matrix_order_t order);

  virtual void
    vector(void * valuePtr, WfCst::WfDataType valueType, unsigned int size);

  virtual void
  voidElement();

  virtual void
  error();

protected:
  bool  firstElt;
};

#endif  // _WFDATAWRITER_HH_
