/**
 * @file  FloodRequestList.cc
 *
 * @brief  Register the state of a request which is send through the MAs graph
 *
 * @author  Sylvain DAHAN (Sylvain.Dahan@lifc.univ-fcomte.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "FloodRequestsList.hh"

#ifdef HAVE_MULTI_MA

#include <cassert>
#include <set>
#ifndef __WIN32__
#include <unistd.h>
#else
#include <Winsock2.h>
#include <windows.h>
#endif
#include "debug.hh"
#include "DIET_compat.hh"

bool
FloodRequestsList::put(FloodRequest &floodRequest) {
  mutex.lock();
  RequestID reqId = floodRequest.getId();
  iterator iter = requestsList.find(reqId);
  bool result = (iter == requestsList.end());
  if (result) {
    requestsList[reqId] = &floodRequest;
  }
  mutex.unlock();
  return result;
} // put

FloodRequest &
FloodRequestsList::get(const RequestID &reqID) {
  int lp = 0;
  FloodRequest *result = NULL;
  bool find = false;
  // waits a maximum of one second
  while (!find && lp < 100) {
    mutex.lock();
    iterator iter = requestsList.find(reqID);
    find = (iter != requestsList.end());
    if (find) {
      result = &(*iter->second);
      requestsList.erase(iter);
      mutex.unlock();
    } else {
      // if the request is not found, wait 10 ms that a thread free
      // the resource access
      mutex.unlock();
      TRACE_TEXT(20, "FloodRequestsLists sleep 10ms\n");
      diet::usleep(10000);
    }
  }
  if (!find) {
    throw FloodRequestNotFoundException(reqID);
  }
  return *result;
} // get

#endif  // HAVE_MULTI_MA
