/**
 * @file  CltWfMgr.hh
 *
 * @brief   Client workflow manager class
 *
 * @author   Abdelkader AMAR (Abdelkader.Amar@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef _CLTWFMGR_HH_
#define _CLTWFMGR_HH_

// STL headers
#include <map>
#include <string>
#include <ostream>
#ifdef __WIN32__
#include<Winsock2.h>
#endif


// IDL headers
#include "CltMan.hh"
#include "MasterAgent.hh"
#include "MaDag.hh"
#ifdef USE_LOG_SERVICE
#include "DietLogComponent.hh"
#endif
#include "WfLogService.hh"


// DIET headers
#include "DIET_data.h"
#include "DIET_client.h"
#include "marshalling.hh"

// Forwarder part
#include "Forwarder.hh"
#include "CltManFwdr.hh"
#ifdef WIN32
	 #define DIET_API_LIB __declspec(dllexport)
#else
	 #define DIET_API_LIB
#endif

#include "events/EventObserver.hh"
#include "WfLogDispatcher.hh"

class NodeSet;
class FWorkflow;
class Dag;
class MetaDag;

class DIET_API_LIB CltWfMgr
: public POA_CltMan
, public PortableServer::RefCountServantBase
, public Singleton<CltWfMgr>
{
	friend Singleton<CltWfMgr>;

	public:
		/**
	 	* Executes a node on a specified Sed (CORBA method)
	 	* @param node_id node identifier
	 	* @param dag_id  dag identifier
	 	* @param sed     SeD where execute service
	 	* @param reqID   request ID (from submit request)
	 	* @param ev      Estimation vector for this SeD (from submit request)
	 	*/
		virtual CORBA::Long
		execNodeOnSed(
			const char*         node_id,
			const char*         dag_id,
			const char*         sed,
			const CORBA::ULong  reqID,
			corba_estimation_t& ev
		);

		/**
		 * Executes a node without specifying the Sed (CORBA method)
		 *
		 * @param node_id node identifier
		 * @param dag_id  dag identifier
		 */
		virtual CORBA::Long
		execNode(const char *node_id, const char *dag_id);

		/**
		 * (CORBA method)
		 * Release the waiting semaphore. This method is used by the MA DAG when workflow execution
		 * is done by this agent
		 */
		virtual char*
		release(const char* dag_id, bool successful);

		/**
		 * Debug function
		 */
		virtual CORBA::Long
		ping();

		/**
		 * Give access to unique reference of CltWfMgr
		 */
		static CltWfMgr&
		init(); // [overload]

		/**
		 * Deletes the unique reference of CltWfMgr
		 */
		void
		terminate();

		/**
		 * Set the MaDag reference
		 *
		 * @param maDagRef MaDag reference
		 */
		void
		setMaDag(MaDag_var maDagRef);

		/**
		 * Set the Master Agent reference
		 *
		 * @param ma Master Agent reference
		 */
		void
		setMA(MasterAgent_var ma);
#ifdef USE_LOG_SERVICE
		/**
		 * Set the log service reference
		 *
		 * @param logComponent log service reference
		 */
		void
		setLogComponent(DietLogComponent *logComponent);
#endif
		/**
		 * Set the workflow log service (GUI)
		 * @param logService wf log service reference
		 */
		void
		setWfLogService(WfLogService_var logService);

		/**
		 * Get a new workflow request ID (for multi-dag submit)
		 */
		CORBA::Long
		getNewWfReqID();

		/**
		 * Execute a dag using the MA DAG.
		 * (if the wfReqID is set in the profile, then this call is part of several
		 * interdependent dag calls)
		 * @param profile dag description
		 */
		diet_error_t
		wfDagCall(diet_wf_desc_t *profile);

		/**
		 * Execute a functional workflow using the MA DAG
		 * @param profile workflow description
		 */
		diet_error_t
		wfFunctionalCall(diet_wf_desc_t *profile);

		/**
		 * Cancel a running dag. This method returns immediately as it only set the dag's status.
		 * Dag will continue running until currently running nodes are finished, and then
		 * MaDag will send the release() call.
		 */
		diet_error_t
		cancelDag(const char *dagId);

		/**
		 * Display all results from a dag
		 */
		diet_error_t
		printAllDagResults(diet_wf_desc_t *profile, std::ostream& stream=std::cout);

		/**
		 * DIsplay all results from a functional wf
		 */
		diet_error_t
		printAllFunctionalWfResults(diet_wf_desc_t *profile, std::ostream& stream=std::cout);

		/**
		 * Read the workflow/dag execution transcript from a file (XML)
		 * (file name is stored in profile)
		 * If file does not exist, method does nothing
		 */
		diet_error_t
		readWorkflowExecutionTranscript(diet_wf_desc_t *profile);

		/**
		 * Store the workflow/dag execution transcript in a file (XML)
		 * (file name is stored in profile)
		 * (If file contains sth, content is overwritten)
		 */
		diet_error_t
		saveWorkflowExecutionTranscript(diet_wf_desc_t *profile, const char *transcriptFileName);

		/**
		 * Store the workflow data file (XML format) after workflow execution.
		 * This file can be used as input for another execution of a workflow
		 * to avoid re-uploading the data items to the DIET platform (the data
		 * IDs are written as attributes of the data items)
		 * (if file already exists, content will be overwritten)
		 */
		diet_error_t
		saveWorkflowDataFile(diet_wf_desc_t *profile, const char *dataFileName);

		/**
		 * Get a scalar result from a dag
		 */
		int
		getWfOutputScalar(diet_wf_desc_t *profile, const char *id, void **value);

		/**
		 * Get a string result from a dag
		 */
		int
		getWfOutputString(diet_wf_desc_t *profile, const char *id, char **value);

		/**
		 * Get a file result from a dag
		 */
		int
		getWfOutputFile(diet_wf_desc_t *profile, const char *id, size_t *size, char **path);

		/**
		 * Get a matrix result from a dag
		 */
		int
		getWfOutputMatrix(
			diet_wf_desc_t*      profile,
			const char*          id,
			void**               value,
			size_t*              nb_rows,
			size_t*              nb_cols,
			diet_matrix_order_t* order
		);

		/**
		 * Get a container result from a dag
		 */
		int
		getWfOutputContainer(diet_wf_desc_t *profile, const char *id, char **dataID);

		/**
		 * Get sink results from a functional workflow
		 * @return dataID contains the DAGDA ID of a container containing all results
		 */
		int
		getWfSinkContainer(diet_wf_desc_t *profile, const char *id, char **dataID);

		/**
		 * terminate a workflow session and free the memory
		 *
		 * @param profile profile of workflow to execute
		 */
		void
		wf_free(diet_wf_desc_t *profile);

	private:
		std::string name;
		/***************************************************************************/
		/*                           PRIVATE methods                               */
		/***************************************************************************/

		/**
		 * Private constructor
		 */
		explicit
		CltWfMgr(const std::string &name);

		/**
		 * Get current time (in milliseconds)
		 */
		double
		getCurrTime();

		/**
		 * Return the object IOR
		 */
		/*const char* myIOR();*/
		const std::string &
		myName() const;

		/**
		 * Return the DAG with a given identifier
		 * @param dag_id Dag identifier
		 * @return dag pointer or NULL if not found
		 */
		Dag*
		getDag(std::string dag_id);

		/**
		 * Initialize status for functional workflow
		 */
		void
		initDagStatus(FWorkflow *wf);

		/**
		 * Set the dag status regarding dags sent for a given functional workflow
		 */
		void
		setWfSubmissionComplete(FWorkflow *wf);

		/**
		 * Get the status regarding dags sent for a given functional workflow
		 */
		bool
		isWfSubmissionComplete(FWorkflow *wf);

		/**
		 * Common part of node execution
		 */
		CORBA::Long
		execNodeCommon(
			const char*         node_id,
			const char*         dag_id,
			const char*         sed,
			const CORBA::ULong  reqID,
			corba_estimation_t& ev
		);

		/**
		 * Common part of the dag submission
		 * @param dagProfile  the diet profile containing the XML code for the Dag
		 *                    (to be sent to MaDag) and eventually the wfReqId
		 * @param dag a ref to a dag (may be NULL if parse is true)
		 * @param parse will parse the XML if set to true
		 * @param release will close the wfReqId on MaDag side if set to true
		 */
		diet_error_t
		wfDagCallCommon(diet_wf_desc_t *dagProfile, Dag *dag, bool parse, bool release);

		/***************************************************************************/
		/*                          PRIVATE attributes                             */
		/***************************************************************************/

		/**
		 * Default data file name
		 */
		static std::string defaultDataFileName;

		/**
		 * MaDag CORBA object reference
		 */
		MaDag_var myMaDag;

		/**
		 * Master Agent reference
		 */
		MasterAgent_var myMA;

#ifdef USE_LOG_SERVICE
		/**
		 * Log service reference
		 */
		DietLogComponent *myLC;
#endif

		/**
		 * Wf Log Service Ref
		 */
		WfLogService_var myLS;

		/**
		 * Wf Event observer
		 */
		WfLogDispatcher *myWfLogDispatcher;

		/**
		 * Event logger object
		 */
		events::EventObserver *myEL;


		/**
		 * Local workflow request ID counter
		 * (different from wf request ID on MaDag)
		 */
		int cltWfReqId;

		/**
		 * Map for profiles and their dags or workflows
		 */
		std::map<diet_wf_desc_t *, NodeSet *> myProfiles;

		/**
		 * Map for metadags
		 */
		std::map<CORBA::Long, MetaDag *> myMetaDags;

		/**
		 * Dags status for functional workflows
		 */
		std::map<FWorkflow *, bool> allDagsSent;

		/**
		 * Dag sent counter
		 */
		int dagSentCount;

		/**
		 * Critical section
		 */
		omni_mutex myLock;

		/**
		 * Synchronisation semaphores
		 */
		omni_semaphore mySem;

		bool instanciationPending;
		/**
		 * Reference time
		 */
		struct timeval refTime;
};

/* Forwarder part. */
class DIET_API_LIB CltWfMgrFwdr : public POA_CltManFwdr,
public PortableServer::RefCountServantBase {
public:
	CltWfMgrFwdr(Forwarder_ptr fwdr, const char *objName);

	virtual CORBA::Long
	execNodeOnSed(const char *node_id, const char *dag_id,
								const char *sed, const CORBA::ULong reqID,
								corba_estimation_t &ev);

	virtual CORBA::Long
	execNode(const char *node_id, const char *dag_id);

	virtual char *
	release(const char *dag_id, bool successful);

	virtual CORBA::Long
	ping();

private:
	Forwarder_ptr forwarder;
	char *objName;
};



#endif   /* not defined _CLTWFMGR._HH */
