/**
 * @file  AccessController.cc
 *
 * @brief  Interface for limiting access to a resource to a configurable consumers
 *
 * @author  - Holly DAIL (Holly.Dail@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "AccessController.hh"

#include <cstdio>
#include <iostream>
using namespace std;
#include <cstdlib>

#include "omnithread.h"
#include "debug.hh"

/** The trace level. */
extern unsigned int TRACE_LEVEL;

#define ACCESS_TRACE_FUNCTION(formatted_text)           \
  TRACE_TEXT(TRACE_ALL_STEPS, "AccessControl::");       \
  TRACE_FUNCTION(TRACE_ALL_STEPS, formatted_text)

AccessController::AccessController(int initialResources) {
  if (initialResources >= 0) {
    this->resourcesInitial = initialResources;
  } else {
    this->resourcesInitial = 0;
    fprintf(stderr, "AccessController:: invalid initialResources of %d.\n",
            initialResources);
    fprintf(stderr, "AccessController:: setting initialResources to 0.\n");
  }

  resourceSemaphore = new omni_semaphore(this->resourcesInitial);
  numFreeSlots = this->resourcesInitial;
  numWaiting = 0;

  reqCounter = 0;
  maxIDReleased = -1;
}

void
AccessController::waitForResource() {
  int myReqID = -1;

  this->globalLock.lock();      /** LOCK */
  myReqID = this->reqCounter++;
  this->numWaiting++;
  this->globalLock.unlock();    /** UNLOCK */

  TRACE_TEXT(TRACE_ALL_STEPS,
             "Thread " << (omni_thread::self())->id()
                       << " / Request " << myReqID <<
             " enqueued ("
                       << this->numWaiting << " waiting, "
                       << this->numFreeSlots <<
             " slots free)" << endl);

  this->resourceSemaphore->wait();

  // TODO: On systems tested semaphore gives FIFO ordering and the following
  // test succeeds only in the case that tasks switch order between the receipt
  // of request IDs and the wait on the semaphore.  This case is not very
  // important because those tasks arrive at almost the same moment.
  // However, semaphores do not guarantee FIFO ordering so for portability
  // another solution must be found to guarantee FIFO.
  if (myReqID != (this->maxIDReleased + 1)) {
    WARNING(
      "Thread " << (omni_thread::self())->id()
                << " / Request " << myReqID <<
      " exiting queue out-of-order.");
  }

  if (this->numFreeSlots <= 0) {
    fprintf(stderr,
            "AccessController:: confusion between "
            "semaphore and numFreeSlots ...");
  }
  if (this->numWaiting <= 0) {
    fprintf(stderr,
            "AccessController:: Unexplained problem "
            "counting waiting threads.");
  }

  this->globalLock.lock();      /** LOCK */
  this->numFreeSlots--;
  this->numWaiting--;
  if (myReqID > this->maxIDReleased) {
    maxIDReleased = myReqID;
  }
  this->globalLock.unlock();    /** UNLOCK */

  TRACE_TEXT(TRACE_ALL_STEPS,
             "Thread " << (omni_thread::self())->id()
                       << " / Request " << myReqID <<
             " got resource ("
                       << this->numWaiting << " waiting, "
                       << this->numFreeSlots <<
             " slots free)" << endl);
} // waitForResource

void
AccessController::releaseResource() {
  this->globalLock.lock();      /** LOCK */
  this->numFreeSlots++;
  this->globalLock.unlock();    /** UNLOCK */

  TRACE_TEXT(TRACE_ALL_STEPS,
             "Thread " << (omni_thread::self())->id()
                       << " released resource ("
                       << this->numWaiting << " waiting, "
                       << this->numFreeSlots <<
             " slots free)" << endl);

  this->resourceSemaphore->post();
} // releaseResource

int
AccessController::getFreeSlotCount() {
  return this->numFreeSlots;
}

int
AccessController::getNumWaiting() {
  return this->numWaiting;
}

int
AccessController::getTotalResourceCount() {
  return this->resourcesInitial;
}
