/**
 * @file MaDag_impl.hh
 *
 * @brief  The MA DAG CORBA object implementation header
 *
 * @author  Abdelkader AMAR (Abdelkader.Amar@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */



#ifndef _MADAG_IMPL_HH_
#define _MADAG_IMPL_HH_

#include <map>
#include <string>

#include "MaDag.hh"
#include "MasterAgent.hh"
#include "WfScheduler.hh"
#include "MultiWfScheduler.hh"
#include "CltMan.hh"
#ifdef USE_LOG_SERVICE
#include "DietLogComponent.hh"
#endif
#include "DagWfParser.hh"
#include "MetaDag.hh"

#include "Forwarder.hh"
#include "MaDagFwdr.hh"
#ifdef WIN32
#define DIET_API_LIB __declspec(dllexport)
#else
#define DIET_API_LIB
#endif

class DIET_API_LIB MaDag_impl
: public POA_MaDag
, public PortableServer::RefCountServantBase
{
	public:
		enum MaDagSchedType {BASIC, GHEFT, GAHEFT, FOFT, SRPT, FCFS};

		MaDag_impl(const char *name,
		           const MaDagSchedType schedType = BASIC,
		           const int interRoundDelay = -1);  // use default

		virtual ~MaDag_impl();

		/**
		 * DAG Workflow processing
		 * Manages the scheduling and orchestration of the submitted DAG
		 * Delegates the execution of each individual task to the client
		 * workflow mgr
		 *
		 * @param  dag_desc   workflow textual description
		 * @param  cltMgrRef  client workflow manager reference
		 * @param  wfReqId    submitted workflow identifier (obtained by
		 * getWfReqId function)
		 * @return the dag Id if workflow submission succeed, -1 if not
		 */
		virtual CORBA::Long
		processDagWf(const corba_wf_desc_t &dag_desc,
		             const char *cltMgrRef,
		             CORBA::Long wfReqId);
		/**
		 * Multi DAG Workflow processing
		 * Manages the scheduling and orchestration of the submitted DAG
		 * when this DAG is linked to other DAGs from the same wf request.
		 * Delegates the execution of each individual task to the client
		 *
		 * @param  dag_desc     workflow textual description
		 * @param  cltMgrRef    client workflow manager reference
		 * @param  wfReqId      submitted workflow identifier (obtained by
		 * getWfReqId function)
		 * @param  release      if false, does not destroy metadag when this request ends
		 * @return the dag Id if workflow submission succeed, -1 if not
		 */
		virtual CORBA::Long
		processMultiDagWf(const corba_wf_desc_t &dag_desc,
		                  const char *cltMgrRef,
		                  CORBA::Long wfReqId,
		                  CORBA::Boolean release);

		/**
		 * Multi DAG release
		 * Free all ressources after the multi-dag is completed
		 */
		virtual void
		releaseMultiDag(CORBA::Long wfReqId);

		/**
		 * DAG cancellation method (non-blocking)
		 * Will stop execution of the dag as soon as all running dag nodes are completed.
		 * (running nodes cannot be stopped asynchronously)
		 * This method returns immediately. The MaDag will call release() on the CltWfMgr
		 * when the last running node is completed.
		 * @param  dagId  id of the dag (provided by MaDag at submission)
		 */
		virtual void
		cancelDag(CORBA::Long dagId);

		/**
		 * Get a new workflow request identifier
		 * @return a new identifier to be used for a wf request
		 */
		virtual CORBA::Long
		getWfReqId();

		/**
		 * Get the client mgr for a given dag
		 * @param dagId the dag identifier
		 * @return client manager pointer (CORBA)
		 */
		virtual CltMan_ptr
		getCltMan(const std::string &dagId);

		/** Used to test if it is alive. */
		virtual CORBA::Long
		ping();

		/**
		 * Set the platform type
		 */
		virtual void
		setPlatformType(MaDag::pfmType_t pfmType);

#ifdef USE_LOG_SERVICE
		/**
		 * Ptr to the DietLogComponent. This ptr can be NULL, so it has to
		 * be checked every time it is used. If it is NULL, no monitoring
		 * messages have to be sent.
		 */
		DietLogComponent *dietLogComponent;

		/**
		 *  Get the DietLogComponent
		 */
		DietLogComponent*
		getDietLogComponent();
#endif
		/**
		 * Get the MA
		 */
		MasterAgent_var
		getMA() const;

	protected:
		/**
		 * set the client manager for a wf request
		 */
		void
		setCltMan(CORBA::Long wfReqId, CltMan_ptr cltMan);

		/**
		 * set the wf request id for a given dag
		 */
		void
		setWfReq(CORBA::Long dagId, CORBA::Long wfReqId);

		/**
		 * Common part of dag processing call methods
		 */
		CORBA::Long
		processDagWfCommon(const corba_wf_desc_t &dag_desc,
		                   const char *cltMgrRef,
		                   CORBA::Long wfReqId,
		                   MetaDag *mDag = nullptr);

		/**
		 * Parse a new dag provided in xml text and create a dag object
		 * @param wf_desc   workflow string description
		 * @param dagId     the dag identifier
		 * @param mDag      ref to a metadag (optional) in case dag is linked to other dags
		 * @return pointer to dag structure (to be destroyed by the caller)
		 */
		Dag*
		parseNewDag(const corba_wf_desc_t &wf_desc,
		            const std::string &dagId,
		            MetaDag *mDag = nullptr)
		noexcept(false);

#ifdef USE_LOG_SERVICE
	/**
	 * setup the DietLogComponent
	 */
	void
	setupDietLogComponent();
#endif

private:
	/**
	 * The Ma Dag name (used for CORBA naming service binding)
	 */
	std::string myName;

	/**
	 * The master agent reference
	 */
	MasterAgent_var myMA;

	/**
	 * The mapping table dagId => wfReqId
	 */
	std::map<std::string, CORBA::Long> wfReqs;

	/**
	 * The mapping table wfReqId => cltManager
	 */
	std::map<CORBA::Long, CltMan_ptr> cltMans;

	/**
	 * The mapping table wfReqId => MetaDag
	 */
	std::map<CORBA::Long, MetaDag *> myMetaDags;

	/**
	 * The meta-scheduler (used for multiworkflow support)
	 */
	madag::MultiWfScheduler *myMultiWfSched;

	/**
	 * Lock to prevent concurrent access
	 */
	omni_mutex myMutex;

	/**
	 * Dag identifier counter
	 */
	CORBA::Long wfReqIdCounter;

	/**
	 * Dag counter
	 */
	CORBA::Long dagIdCounter;
};

class DIET_API_LIB MaDagFwdrImpl
: public POA_MaDag
, public PortableServer::RefCountServantBase
{
	public:
		MaDagFwdrImpl(Forwarder_ptr fwdr, const char *objName);

		virtual CORBA::Long
		processDagWf(const corba_wf_desc_t &dag_desc,
		             const char *cltMgrRef,
		             CORBA::Long wfReqId);

		virtual CORBA::Long
		processMultiDagWf(const corba_wf_desc_t &dag_desc,
		                  const char *cltMgrRef,
		                  CORBA::Long wfReqId, CORBA::Boolean release);

		virtual CORBA::Long
		getWfReqId();

		virtual void
		releaseMultiDag(CORBA::Long wfReqId);

		virtual void
		cancelDag(CORBA::Long dagId);

		virtual void
		setPlatformType(MaDag::pfmType_t pfmType);

		virtual CORBA::Long
		ping();

	protected:
		Forwarder_ptr forwarder;
		char *objName;
};

#endif   /* not defined _MADAG_IMPL_HH */
