/**
 * @file BatchCreator.cc
 *
 * @brief  Batch System Management with performance prediction
 *
 * @author  Yves Caniou (yves.caniou@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

#include "deltacloud_config.h"

// #include <cstring>
#include "OAR1_6BatchSystem.hh"
#include "OAR2_XBatchSystem.hh"
#include "Loadleveler_BatchSystem.hh"
#include "PBS_BatchSystem.hh"
#include "Slurm_BatchSystem.hh"

#ifdef DIET_USE_EC2CLOUD
#include "Eucalyptus_BatchSystem.hh"
#endif

#ifdef DIET_USE_DELTACLOUD
#include "Deltacloud_BatchSystem.hh"
#endif

#include "BatchCreator.hh"

#include <cstring>

#include "debug.hh"

const char *const BatchCreator::batchNames[NUMBER_OF_SUPPORTED_BATCH] = {
  "oar1.6",
  "loadleveler",
  "sge",
  "pbs",
  "oar2.x",
  "slurm"

#ifdef DIET_USE_EC2CLOUD
  , "eucalyptus"
#endif
#ifdef DIET_USE_DELTACLOUD
  , "deltacloud"
#endif
};

// "shellscript",
// "condor", "dqs", "loadleveler", "lsf", "pbs", "sge",
// "oar1.6"

BatchCreator::BatchCreator() {
}

BatchSystem *
BatchCreator::getBatchSystem(const char *batchName) {
  batchID_t batchID;

  if (!(existBatchScheduler(batchName, &batchID))) {
    return NULL;
  }

  switch ((int) batchID) {
  case OAR1_6:
    return new OAR1_6BatchSystem(batchID, batchNames[batchID]);

  case LOADLEVELER:
    return new Loadleveler_BatchSystem(batchID, batchNames[batchID]);

  case PBS:
    return new PBS_BatchSystem(batchID, batchNames[batchID]);

  case OAR2_X:
    return new OAR2_XBatchSystem(batchID, batchNames[batchID]);

  case SLURM:
    return new Slurm_BatchSystem(batchID, batchNames[batchID]);

#ifdef DIET_USE_EC2CLOUD
  case EUCALYPTUS:
    return new Eucalyptus_BatchSystem(batchID, batchNames[batchID]);

#endif

#ifdef DIET_USE_DELTACLOUD
  case DELTACLOUD:
    TRACE_TEXT(TRACE_MAIN_STEPS, "Deltacloud USED")
    return new Deltacloud_BatchSystem(batchID, batchNames[batchID]);
#endif

  default:
    return NULL;
  } // switch
} // getBatchSystem

int
BatchCreator::existBatchScheduler(const char *batchName,
                                  batchID_t *batchID) {
  int id_tmp = 0;

  while ((id_tmp < NUMBER_OF_SUPPORTED_BATCH)
         && (strncmp(batchName, batchNames[id_tmp], 4) != 0)) {
    /* for the moment, only a check on the 4 first chars is needed */
    id_tmp++;
  }

#ifdef YC_DEBUG
  TRACE_TEXT(TRACE_MAIN_STEPS, "YC_DEBUG: "
             << "NUMBER_OF_SUPPORTED_BATCH: " << NUMBER_OF_SUPPORTED_BATCH
             << endl);
  TRACE_TEXT(TRACE_MAIN_STEPS, "YC_DEBUG: "
             << "; BatchID: " << id_tmp
             << "; batchName: " << batchName
             << endl);
  TRACE_TEXT(TRACE_MAIN_STEPS, "YC_DEBUG: "
             << "; batchNames[batchID]: " << batchNames[id_tmp]
             << "; cmp(): " << strncmp(batchName, batchNames[id_tmp], 4)
             << endl << endl << endl);
#endif // ifdef YC_DEBUG

  *batchID = (batchID_t) id_tmp;

  if (id_tmp == NUMBER_OF_SUPPORTED_BATCH) {
    return 0;
  }
  return 1;
} // existBatchScheduler
