/**
 * @file Agent.cc
 *
 * @brief  DIET forwarder implementation - Agent forwarder implementation
 *
 * @author  Gael Le Mahec   (gael.le.mahec@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "DIETForwarder.hh"
#include "ORBMgr.hh"
#include <string>
#include <iostream>
#include <stdio.h>

::CORBA::Long
DIETForwarder::agentSubscribe(const char *agentName,
                              const char *hostname,
                              const ::SeqCorbaProfileDesc_t &services,
                              const char *objName) {
  std::string objString(objName);
  std::string name;

  if (!remoteCall(objString)) {
    return getPeer()->agentSubscribe(agentName, hostname,
                                     services, objString.c_str());
  }

  name = getName(objString);

  Agent_var agent =
    ORBMgr::get().resolve<Agent, Agent_var>(AGENTCTXT, name, this->name);
  return agent->agentSubscribe(agentName, hostname, services);
} // agentSubscribe

::CORBA::Long
DIETForwarder::serverSubscribe(const char *seDName, const char *hostname,
                               const ::SeqCorbaProfileDesc_t &services,
                               const char *objName) {
  std::string objString(objName);
  std::string name;

  if (!remoteCall(objString)) {
    return getPeer()->serverSubscribe(seDName, hostname,
                                      services, objString.c_str());
  }
  name = getName(objString);

  Agent_var agent =
    ORBMgr::get().resolve<Agent, Agent_var>(AGENTCTXT, name, this->name);
  return agent->serverSubscribe(seDName, hostname, services);
} // serverSubscribe

::CORBA::Long
DIETForwarder::childUnsubscribe(::CORBA::ULong childID,
                                const ::SeqCorbaProfileDesc_t &services,
                                const char *objName) {
  std::string objString(objName);
  std::string name;

  if (!remoteCall(objString)) {
    return getPeer()->childUnsubscribe(childID, services, objString.c_str());
  }

  name = getName(objString);

  Agent_var agent =
    ORBMgr::get().resolve<Agent, Agent_var>(AGENTCTXT, name, this->name);
  return agent->childUnsubscribe(childID, services);
} // childUnsubscribe

::CORBA::Long
DIETForwarder::childRemoveService(::CORBA::ULong childID,
                                  const ::corba_profile_desc_t &profile,
                                  const char *objName) {
  std::string objString(objName);
  std::string name;

  if (!remoteCall(objString)) {
    return getPeer()->childRemoveService(childID, profile, objString.c_str());
  }

  name = getName(objString);

  Agent_var agent =
    ORBMgr::get().resolve<Agent, Agent_var>(AGENTCTXT, name, this->name);
  return agent->childRemoveService(childID, profile);
} // childRemoveService

::CORBA::Long
DIETForwarder::addServices(::CORBA::ULong myID,
                           const ::SeqCorbaProfileDesc_t &services,
                           const char *objName) {
  std::string objString(objName);
  std::string name;

  if (!remoteCall(objString)) {
    return getPeer()->addServices(myID, services, objString.c_str());
  }

  name = getName(objString);

  Agent_var agent =
    ORBMgr::get().resolve<Agent, Agent_var>(AGENTCTXT, name, this->name);
  return agent->addServices(myID, services);
} // addServices

void
DIETForwarder::getResponse(const ::corba_response_t &resp,
                           const char *objName) {
  std::string objString(objName);
  std::string name;

  if (!remoteCall(objString)) {
    return getPeer()->getResponse(resp, objString.c_str());
  }

  name = getName(objString);

  Agent_var agent =
    ORBMgr::get().resolve<Agent, Agent_var>(AGENTCTXT, name, this->name);
  return agent->getResponse(resp);
} // getResponse

char *
DIETForwarder::getDataManager(const char *objName) {
  std::string objString(objName);
  std::string name;

  if (!remoteCall(objString)) {
    return getPeer()->getDataManager(objString.c_str());
  }

  name = getName(objString);

  Agent_var agent =
    ORBMgr::get().resolve<Agent, Agent_var>(AGENTCTXT, name, this->name);
  return agent->getDataManager();
} // getDataManager

SeqString *
DIETForwarder::searchData(const char *request,
                          const char *objName) {
  std::string objString(objName);
  std::string name;

  if (!remoteCall(objString)) {
    return getPeer()->searchData(request, objString.c_str());
  }

  name = getName(objString);

  Agent_var agent =
    ORBMgr::get().resolve<Agent, Agent_var>(AGENTCTXT, name, this->name);
  return agent->searchData(request);
} // searchData
