/**
* @file  WfExpression.hh
* 
* @brief  The class used to parse and execute conditional expressions used in workflow control structures
* 
* @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr)
* 
* @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/


#ifndef _WFEXPRESSION_HH_
#define _WFEXPRESSION_HH_

#include <list>
#include <string>
#include <xqilla/xqilla-simple.hpp>
#include "WfUtils.hh"


/*****************************************************************************/
/*                       WfExpressionParser class                            */
/*****************************************************************************/

class WfExpressionParser {
public:
  /**
   * Parse a string containing a XQuery and returns the XQilla query
   * object (query is not executed yet)
   * @param queryStr  string containing a XQuery
   */
  XQQuery*
  parse(const std::string& queryStr);

  /**
   * Returns the unique instance of this class
   */
  static WfExpressionParser*
  instance();

  /**
   * Destructor
   */
  ~WfExpressionParser();

private:

  WfExpressionParser();

  /**
   * Unique instance reference
   */
  static WfExpressionParser* myInstance;

  /**
   * Xqilla environment object
   */
  XQilla* myImpl;
};

/*****************************************************************************/
/*                        WfExprVariable class                               */
/*****************************************************************************/

class WfExprVariable {
public:
  /**
   * Initialize a variable
   * @param varName string containing the variable name
   * @param varType type of the variable
   */
  WfExprVariable(const std::string& varName, const WfCst::WfDataType varType);

  virtual ~WfExprVariable() {
  }

  /**
   * Get name
   */
  const std::string&
  getName() { return myName; }

  /**
   * Get value
   */
  const std::string&
  getValue() { return myValue; }


  const WfCst::WfDataType&
  getType() {return myType; }


  /**
   * Generate the XQuery declaration of the variable
   * @param output  an output stream
   */
  virtual void
  getXQDeclaration(std::ostream& output);

  /**
   * Set the value of the variable
   * @param varValue raw value if scalar, XML-encoded if container
   */
  virtual void
  setValue(const std::string& varValue);

  /**
   * Set a default value for the variable
   */
  virtual void
  setDefaultValue();

protected:
  std::string myName;
  WfCst::WfDataType myType;
  std::string myValue;
};

/*****************************************************************************/
/*                          WfExpression class                               */
/*****************************************************************************/

class WfExpression {
public:
  WfExpression();
  virtual ~WfExpression();

  /**
   * Set the conditional expression
   * @param exprStr XQuery containing variables with syntax $var
   */
  virtual void
  setExpression(const std::string& exprStr);

  /**
   * Add a variable to the expression
   * @param var ref to a WfExprVariable
   */
  virtual void
  addVariable(WfExprVariable*  var);

  /**
   * Get the expression
   */
  const std::string&
  getExpression();

  /**
   * Get the full query (used for exceptions)
   */
  const std::string&
  getQueryString();

  /**
   * Check if a variable name is used in the expression
   */
  // characters that can happen after a variable
  static std::string XQVarSeparators;

  bool
  isVariableUsed(const std::string& varName);

  /**
   * Evaluate the expression
   */
  virtual void
  evaluate();

protected:
  /**
   * Method to initialize the XQuery before parsing
   * (uses both the expression and the variables)
   */
  void
  initQuery();

  /**
   * Method to parse an expression containing variables
   */
  void
  parseQuery();

  /**
   * Original expression
   */
  std::string myExpression;

  /**
   * Query (in XQuery lang)
   */
  std::string myQueryStr;

  /**
   * Result
   */
  std::string myResult;

  /**
   * Variables storage
   */
  std::list<WfExprVariable*> myVariables;

  /**
   * Parser
   */
  WfExpressionParser* myParser;

  /**
   * The query object
   */
  XQQuery* myQuery;

};


/*****************************************************************************/
/*                       WfBooleanExpression class                           */
/*****************************************************************************/

class WfBooleanExpression : public WfExpression {
public:
  /**
   * Evaluate the expression
   * @return true if the expression evaluates to 'true', else false
   */
  bool
  testIfTrue();
};

#endif  // _WFEXPRESSION_HH_
