/**
 * @file  KeyString.hh
 *
 * @brief  Hashed string object to be used with container
 *
 * @author  Sylvain DAHAN (Sylvain.Dahan@lifc.univ-fcomte.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef _KEY_STRING_HH_
#define _KEY_STRING_HH_



#include <cstdlib>
#include <iostream>

#include "OSIndependance.hh"

class KeyString {
public:
  /**
   * calculate the hash value of the key
   *
   * @param id the string for which the hash value is calculated. Must
   * be not NULL.
   *
   * @return the hash value of id.
   */
  static u_int32_t  hash(const char *id);

  /**
   * creates a new KeyString.
   *
   * @param value value of the key. It value is copied by the
   * KeyString and the memory used as argument can be free by the
   * user. value must be not NULL.
   */
  KeyString(const char *value);

  /**
   * Creates a clone of the KeyString given in argument.
   *
   * @param masterAgentId the KeyString which is cloned
   */
  KeyString(const KeyString &masterAgentId);

  /**
   * The KeyString is destroyed.
   */
  ~KeyString();

  /**
   * change the value of the key.
   *
   * @param masterAgentId the new value of the key.
   */
  KeyString &
  operator = (const KeyString &masterAgentId);

  /**
   * returnes a char* that containes the value of the
   * KeyString.
   */
  operator const char *() const;


  friend bool
    operator < (const KeyString &a, const KeyString &b);

  friend bool
    operator == (const KeyString &a, const KeyString &b);

  friend std::ostream &
  operator << (std::ostream & output, const KeyString &id);

private:
  /**
   * value of the key.
   */
  char *str;

  /**
   * hash value of the identifier
   */
  u_int32_t hashValue;
};


/**
 * returns true if the representation of a is lesser than the
 * representation of b.
 *
 * @param a a KeyString
 *
 * @param b a KeyString
 */
bool
  operator < (const KeyString &a, const KeyString &b);

/**
 * returns true if a and b represent the same ID.
 *
 * @param a a KeyString
 *
 * @param b a KeyString
 */
bool
  operator == (const KeyString &a, const KeyString &b);

/**
 * Writes an ascii description of an id.
 */
inline std::ostream &
operator << (std::ostream & output, const KeyString &key) {
  return output << key.str;
}


#endif  // _KEY_STRING_HH_
