/**
 * @file  statistics.cc
 *
 * @brief  DIET statistics tools source code
 *
 * @author  Ludovic BERTSCH (Ludovic.Bertsch@ens-lyon.fr)
 *          Eddy CARON (Eddy.Caron@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

#include "stats_config.h"

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <sys/types.h>
#include "debug.hh"
#include "statistics.hh"
#include "OSIndependance.hh"

#if HAVE_STATISTICS

const char *const STAT_TYPE_STRING[] = {
  "IN  ",
  "OUT ",
  "INFO"
};

/** The trace level. */
extern unsigned int TRACE_LEVEL;

#define STAT_TRACE_FUNCTION(formatted_text)             \
  TRACE_TEXT(TRACE_ALL_STEPS, "Statistics::");          \
  TRACE_FUNCTION(TRACE_ALL_STEPS, formatted_text)


static char *STAT_FILE_NAME;  // We don't need it to be global
FILE *STAT_FILE = NULL;
static int USING_STATS = 1;

void
gen_stat(int type, const char *myname, const char *message) {
  if (STAT_FILE != NULL) {
    struct timeval tv;
#ifdef __WIN32__
    struct timeval tz;
#else
	struct timezone tz;
#endif

    if (gettimeofday(&tv, &tz) == 0) {
      fprintf(STAT_FILE, "%10ld.%06ld|%s|[%s] %s\n",
              (long int) tv.tv_sec, (long int) tv.tv_usec,
              STAT_TYPE_STRING[type], myname, message);

      /* Examples of generated trace :
       * 123456.340569|IN  |[Name of DIET component] submission.start
       * 123456.340867|INFO|[Name of DIET component] submission.phase1
       * 123455.345986|INFO|[Name of DIET component] submission.phase2
       * 123456.354032|OUT |[Name of DIET component] submission.end
       */
    }
  }
} /* gen_stat */

void
do_stat_init() {
  if (STAT_FILE != NULL) {
    TRACE_TEXT(TRACE_STRUCTURES,
               "do_stat_init() - Stats module already initialized.\n");
  } else {
    STAT_FILE_NAME = getenv("DIET_STAT_FILE_NAME");

    if (STAT_FILE_NAME != NULL) {
      STAT_FILE = fopen(STAT_FILE_NAME, "a");
      if (STAT_FILE == NULL) {
        TRACE_TEXT(
          TRACE_MAIN_STEPS,
          "Warning: do_stat_init() - Unable to open file " <<
          STAT_FILE_NAME
                                                           << ".\n");
        ERROR_DEBUG("do_stat_init() - Check DIET_STAT_FILE_NAME env variable?", );
      } else {
        TRACE_TEXT(TRACE_ALL_STEPS, "* Statistics collection: enabled "
                   << "(file" << STAT_FILE_NAME << ")\n");
        USING_STATS = 1;
      }
    } else if (USING_STATS) {
      TRACE_TEXT(TRACE_ALL_STEPS, "* Statistics collection: disabled\n");
      TRACE_TEXT(
        TRACE_ALL_STEPS,
        "\tTo collect statistics, set env variable DIET_STAT_FILE_NAME.\n");
      USING_STATS = 0;
    }
  }
} // do_stat_init

void
do_stat_flush() {
  if (USING_STATS) {
    if (STAT_FILE == NULL) {
      TRACE_TEXT(
        TRACE_MAIN_STEPS,
        "Warning (do_stat_finalize): stats module is NOT initialized!\n");
    } else {
      if (fflush(STAT_FILE) != 0) {
        ERROR_DEBUG("do_stat_flush() - Unable to flush stat file.", );
      }
    }
  }
} // do_stat_flush

void
do_stat_finalize() {
  if (USING_STATS) {
    if (STAT_FILE == NULL) {
      TRACE_TEXT(
        TRACE_MAIN_STEPS,
        "Warning (do_stat_finalize): stats module is NOT initialized!\n");
    } else {
      if (fclose(STAT_FILE) < 0) {
        ERROR_DEBUG("do_stat_finalize() - Unable to close stat file.", );
      }
      STAT_FILE = NULL;
    }
  }
} // do_stat_finalize

#endif  // HAVE_STATISTICS
