/**
* @file  WfNode.hh
* 
* @brief  Node description class that contains the i/o ports of a workflow node 
* 
* @author  Abdelkader AMAR (Abdelkader.Amar@ens-lyon.fr)
*          Benjamin ISNARD (benjamin.isnard@ens-lyon.fr)
* 
* @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
*/


#ifndef _NODE_HH_
#define _NODE_HH_

#include <iostream>
#include <list>
#include <map>
#include <set>
#include <string>
#include <vector>
#include "WfUtils.hh"
#include "WfPort.hh"
#ifdef WIN32
#define DIET_API_LIB __declspec(dllexport)
#else
#define DIET_API_LIB
#endif

class NodeSet;

class DIET_API_LIB WfNode  {
public:
  /*********************************************************************/
  /* public methods                                                    */
  /*********************************************************************/

  /**
   * The Node default constructor
   * @param id         the node id
   */
  WfNode(const std::string& id);

  /**
   * Node destructor
   */
  virtual ~WfNode();

  /**
   * To get the node id *
   */
  const std::string&
  getId() const;


  /******************************/
  /* Links with other nodes     */
  /******************************/

  /**
   * initializes the previous nodes using both
   * the control dependencies (\<prec\> tag) and the
   * data dependencies (ports links)
   */
  virtual void
  setNodePrecedence(NodeSet *nodeSet) noexcept(false);

  /**
   * Add a new predecessor
   * (may check some constraints before adding the predecessor effectively)
   * @param node  ptr to the predecessor
   * @param fullNodeId  contains the id of the predecessor eventually prefixed (if external)
   */
  virtual void
  addNodePredecessor(WfNode *node, const std::string& fullNodeId);

  /**
   * return the number of previous nodes
   * (setNodePredecessors must be called before)
   */
  unsigned int
  prevNodesNb() const;

  /**
   * return an iterator on the first previous nodes
   */
  std::vector<WfNode*>::iterator
  prevNodesBegin();

  /**
   * return an iterator on the end of previous nodes
   */
  std::vector<WfNode*>::iterator
  prevNodesEnd();

  /**
   * return the number of next nodes
   */
  unsigned int
  nextNodesNb() const;

  /**
   * return an iterator on the first next node
   */
  std::list<WfNode*>::iterator
  nextNodesBegin();

  /**
   * return an iterator on the end of next nodes
   */
  std::list<WfNode*>::iterator
  nextNodesEnd();

  /**
   * Link the ports by object references
   */
  virtual void
  connectNodePorts() noexcept(false);

  /**
   * return true if the node is an input node *
   * only the nodes with no previous node are considered as dag input  *
   */
  virtual bool
  isAnInput() const;

  /**
   * return true if the node is an output node *
   * only the nodes with no next node are considered as dag exit  *
   */
  virtual bool
  isAnExit() const;

  /******************************/
  /* PORTS                      */
  /******************************/

  /**
   * create and add a new port to the node *
   * @param id        the port identifier
   * @param ind       the port index in diet profile
   * @param portType  the port type (in, out, inout)
   * @param dataType  the data type (constant defined in WfCst class)
   * @param depth     the depth of the list structure (0 if no list)
   */
  virtual WfPort *
  newPort(std::string id,
          unsigned int ind,
          WfPort::WfPortType portType,
          WfCst::WfDataType dataType,
          unsigned int depth) noexcept(false) = 0;

  /**
   * check if port already exists
   * @return true if the port id is already defined
   */
  bool
  isPortDefined(const std::string& id);

  /**
   * Get the port by id. If not found returns NULL
   * @param id the port id
   */
  WfPort*
  getPort(const std::string& id) noexcept(false);

  /**
   * Get nb of ports
   */
  unsigned int
  getPortNb() const;

  /**
   * Get nb of ports by type
   */
  unsigned int
  getPortNb(WfPort::WfPortType portType) const;

  /**
   * Get port by index
   */
  const WfPort*
  getPortByIndex(unsigned int portIdx) const;

  /**
   * Get ports description
   * @return a std::string (example: "IN DIET_INT, OUT DIET_DOUBLE")
   */
  std::string
  getPortsDescr() const;

protected:
  /**
   * add a previous node using the node id
   * (this is the only way to add new predecessors; after adding all node
   * ids the method setNodePredecessors is called to convert the
   * predecessors ids into object references)
   */
  void
  addPrevId(const std::string& nodeId);

  /**
   * remove a previous node using the node id
   */
  void
  remPrevId(const std::string& nodeId);

  /**
   * Set the precedence relationship between two nodes
   * ie add the successor to the list of successors of the predecessor
   * and add the predecessor to the list of predecessors of the successor
   * @param index the index of the predecessor (in the list of predecessors)
   * @param node  the predecessor ref
   */
  virtual void
  setPrev(int index, WfNode * node);

  /**
   * Add a new next node reference *
   * @param n the next node reference
   */
  void
  addNext(WfNode * n);

  /**
   * Node id *
   */
  std::string myId;

  /**
   * Method to add a port
   * @param portId  identifier of the port
   * @param port    pointer to the port
   * @return  same pointer as argument
   * @exception WfStructException if duplicate port (pointer is deleted)
   */
  WfPort *
  addPort(const std::string& portId, WfPort* port)
    noexcept(false);

  /**
   * ports map<id, reference> *
   */
  std::map<std::string, WfPort*> ports;

  /**
   * The previous nodes list
   * (use a vector because nb of items is known by prevNodeIds)
   */
  std::vector<WfNode*> prevNodes;

  /**
   * The next nodes list
   * (use a list because nb of items is not known in advance)
   */
  std::list<WfNode*> nextNodes;

private:

  /**
   * the previous nodes ids
   * (use a set to avoid searching duplicate when inserting)
   */
  std::set<std::string> prevNodeIds;
};

#endif   /* not defined _NODE_HH */
