/**
 * @file Cori_Easy_Memory.cc
 *
 * @brief  CoRI Collectors of ressource information (Memory part)
 *
 * @author  Frauenkron Peter (Peter.Frauenkron@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifdef CORI_HAVE_SYS_SYSINFO
#include <sys/sysinfo.h> // get_phys_pages (), get_avphys_pages ()
#endif
#include <cstdio>
#include "Cori_Easy_Memory.hh"
#include <cstdlib> // system()
#include <iostream>
#include <fstream>
#include <cstring>
// #include <string.h>//srtcmp
#include <string>
#include "debug.hh"
#include "OSIndependance.hh" // int getpagesize (void)

using namespace std;

int
Easy_Memory::get_Total_Memory(double *result) {
  double temp1, temp2, temp3;
  if (!get_Info_Memory_byProcMem(&temp1, 0)) {
    *result = temp1;
    return 0;
  } else if (!get_Total_Memory_bysysinfo(&temp1)
             && (!get_Avail_Memory_byvmstat(&temp2))
             && (!get_Avail_Memory_bysysinfo(&temp3))) {
    double pagesize = temp2 / temp3;
    *result = temp1 * pagesize;
    return 0;
  } else {
    *result = 0;
    return 1;
  }
} // get_Total_Memory
int
Easy_Memory::get_Avail_Memory(double *result) {
  double temp;
  if (!get_Info_Memory_byProcMem(&temp, 1)) {
    *result = temp;
    return 0;
  } else if (!get_Avail_Memory_byvmstat(&temp)) {
    *result = temp;
    return 0;
  } else {
    *result = 0;
    return 1;
  }
} // get_Avail_Memory

/****************************************************************************/
/* Private methods                                                          */
/****************************************************************************/

int
Easy_Memory::get_Total_Memory_bysysinfo(double *result) {
  // using  get_phys_pages ()
  // returns the total number of physical pages the system has
  // and
  // int getpagesize (void)
  // Inquire about the virtual memory page size of the machine.
#if defined CORI_HAVE_get_avphys_pages && defined (CORI_HAVE_getpagesize)
  *result = (get_phys_pages() / 1024) * (getpagesize() / 1024);
  return 0;

#else
  // chercher autre solutions
  return 1;

#endif
} // get_Total_Memory_bysysinfo

int
Easy_Memory::get_Avail_Memory_bysysinfo(double *result) {
  // using get_avphys_pages ()
  // returns the number of physical available pages the system has
  // and
  // int getpagesize (void)
  // Inquire about the virtual memory page size of the machine.
#if defined CORI_HAVE_get_avphys_pages && defined (CORI_HAVE_getpagesize)
  *result = get_avphys_pages() * getpagesize() / (1024 * 1024);
  return 0;

#else
  // chercher autres sol
  return 1;

#endif
} // get_Avail_Memory_bysysinfo

int
Easy_Memory::get_Avail_Memory_byvmstat(double *result) {
  int returnval = 1;
#ifdef WIN32
  FILE *myfile = _popen("vmstat", "r");
#else
  FILE *myfile = popen("vmstat", "r");
#endif
  char word[256];
  if ((myfile != NULL)) {
    if (!feof(myfile)) {
      fgets(word, 256, myfile);
      if (!feof(myfile)) {
        fgets(word, 256, myfile);
        int i = 0;
        while ((!feof(myfile)) && (i < 4)) {
          fscanf(myfile, "%255s", word);
          i++;
        }
        if (i == 4) {
          *result = atof(word) / 1024;
          returnval = 0;
        } else {return 1;
        }
      }
    }
#ifdef WIN32
    _pclose(myfile);
#else
    pclose(myfile);
#endif
  }
  return returnval;
} // get_Avail_Memory_byvmstat


/*
   cat /proc/meminfo*/
int
Easy_Memory::get_Info_Memory_byProcMem(double *resultat,
                                       int freemem) {
#ifdef CORI_HAVE_PROCMEM

  char word[256];
  char demanded[256];
  /* looking in the /proc/meminfo data file*/
  ifstream file("/proc/meminfo");

  if (freemem) {
    strcpy(demanded, "MemFree:");
  } else {strcpy(demanded, "MemTotal:");
  }


  if (file.is_open()) {
    while (!file.eof()) { // look at the whole file
      if (strcmp(demanded, word) == 0) {
        file >> word;
        *resultat = atof(word) / 1024;
        return 0;
      }
      file >> word;
    }
    file.close();
  } else {
    TRACE_TEXT(TRACE_MAX_VALUE, "Error on reading file");
    return 1;
  }
#endif  // CORI_HAVE_PROCMEM
  return 1;
} // get_Info_Memory_byProcMem


// int main (){
// Easy_Memory mem;
// double maxmem;
// double availmem;
// if (!mem.get_Total_Memory(&maxmem))
// printf("%d\n", maxmem);
// else printf("no info\n");

// if (!mem.get_Avail_Memory(&availmem))
// printf("%d\n", availmem);
// else printf("no info\n");



// }
