/**
 * @file  SpecificClientScheduler.hh
 *
 * @brief   Specific client scheduling : to implement some specific scheduling
 *
 * @author   Abdelkader AMAR (Abdelkader.Amar@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#ifndef _SPECIFICCLIENTSCHEDULER_HH_
#define _SPECIFICCLIENTSCHEDULER_HH_

#include "SeD.hh"
#include "response.hh"
#include <vector>
#include <string>

class SpecificClientScheduler {
public:
  SpecificClientScheduler();

  virtual ~SpecificClientScheduler();

  static const char *BurstRequest;

  static const char *BurstLimit;

  /**
   * initialize the scheduler
   *
   * @param scheduling_name scheduling name
   */
  static void
  setSchedulingId(const char *scheduling_name);

  static void
  start(SeD_var &chosenServer, corba_response_t *response);

  /**
   * Generic method to execute a specific scheduling.
   *
   * @param scheduling_name the scheduling name defined in
   *       USE_SPECIFIC_SCHEDULING entry in client configuration file
   * @param chosenServer  the SeD reference
   * @param reponse       the received response
   */
  virtual void
  schedule(const char *scheduling_name,
           SeD_var &chosenServer,
           corba_response_t *reponse);

  /**
   * Set the current SpecificScheduler
   */
  static void
  setScheduler(SpecificClientScheduler *scheduler);


  static void
  pre_diet_call();

  static void
  post_diet_call();

  virtual void
  preCall();

  virtual void
  postCall();

  /*
   * Return if the Burst scheduling is enabled
   */
  static bool
  isEnabled();

  /**
   * Utility method
   *
   * @param option option to check
   * @param params parameters vector
   */
  static bool
  isOptionEnabled(std::string option, std::vector<std::string> &params);

  /**
   * Utility method: remove blanks from a string
   *
   * @param token string to trim
   */
  static void
  removeBlanks(std::string &token);

  /**
   * Utility method: split a string given a delimiter
   *
   * @param str the string to split
   * @param delim string delimeter
   * @param results the elements of the split string
   */

  static void
  stringSplit(std::string str, std::string delim,
              std::vector<std::string> &results);

private:
  /**
   * Scheduler state
   */
  static bool enabled;

  /**
   * Scheduler ID
   */
  static std::string schedulerId;

  /**
   * The used scheduler
   */
  static SpecificClientScheduler *myScheduler;

  /**
   * Synchronisation semaphore
   */
  omni_semaphore mySem;

  /**
   * scheduling options
   */
  std::string schedulingOptions;

  /**
   * parameters vector
   */
  std::vector<std::string> myParams;

  /**
   * set the scheduling options
   */
  void
  setSchedulingOptions(const char *schedOptions);

  /**
   * BURST REQUEST : used to implement a better round robbin for
   * following configuration : 1 client -> 1 MA -> n SeDs.
   *
   * @param chosenServer the SeD reference
   * @param response     the MA response
   */
  virtual void
  burstRequest(SeD_var &chosenServer, corba_response_t *response);

  /**
   * BURST LIMIT REQUEST : used to implement a better round robbin for
   * following configuration : 1 client -> 1 MA -> n SeDs.
   * Additionnaly it allow only n simultaneous call
   *
   * @param chosenServer the SeD reference
   * @param response     the MA response
   */
  virtual void
  burstLimitRequest(SeD_var &chosenServer, corba_response_t *response);
};


#endif   /* not defined _SPECIFICCLIENTSCHEDULER._HH */
