/**
 * @file  DataRelationMgr.cc
 *
 * @brief  Class used to manage relationships between data items (used for containers in DAGDA)
 *
 * @author  Benjamin Isnard (benjamin.isnard@ens-lyon.fr
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */


#include "DataRelationMgr.hh"
#include "debug.hh"
#include <iostream>

DataRelationMgr::DataRelationMgr() {
}

void
DataRelationMgr::addRelation(const std::string &dataID1,
                             const std::string &dataID2,
                             long index,
                             long flag) {
  dataRelationValue_t value = {dataID2, index, flag};
  myLock.lock(); /** LOCK */
  myMap.insert(make_pair(dataID1, value));
  myLock.unlock(); /** UNLOCK */
}

void
DataRelationMgr::remRelation(const std::string &dataID1,
                             long index) {
  myLock.lock(); /** LOCK */
  for (std::multimap<std::string, dataRelationValue_t>::iterator iter =
         myMap.lower_bound(dataID1);
       iter != myMap.upper_bound(dataID1);
       ++iter) {
    if (iter->second.index == index) {
      myMap.erase(iter);
      break;
    }
  }
  myLock.unlock(); /** UNLOCK */
} // remRelation

void
DataRelationMgr::remAllRelation(const std::string &dataID, bool reverse) {
  // Delete links FROM the item
  myLock.lock();
  int count = myMap.erase(dataID);
  myLock.unlock();
  TRACE_TEXT(TRACE_ALL_STEPS, "Removed " << count << " relations for key "
                                         << dataID << "\n");
  // Delete links TO the item
  if (reverse) {
    // TODO delete all links TO the item
  }
} // remAllRelation

/**
 * Get the list of relations for a given dataID
 * Note: the SeqString and SeqLong must be pre-allocated by the caller
 */
void
DataRelationMgr::getRelationList(const std::string &dataID,
                                 SeqString &dataIDList,
                                 SeqLong &flagList,
                                 bool ordered) {
  int ix = 0;
  int listSize = dataIDList.length();
  myLock.lock();
  for (std::multimap<std::string, dataRelationValue_t>::iterator iter =
         myMap.lower_bound(dataID);
       iter != myMap.upper_bound(dataID);
       ++iter) {
    int jx = ordered ? iter->second.index : ix++;
    if (jx >= listSize) {
      INTERNAL_ERROR(
        "Mismatching container element index with nb of relationships", 1);
    }
    dataIDList[jx] = CORBA::string_dup(iter->second.ID.c_str());
    flagList[jx] = iter->second.flag;
  }
  myLock.unlock();
} // getRelationList

unsigned int
DataRelationMgr::getRelationNb(const std::string &dataID) {
  unsigned int ix = 0;
  myLock.lock();
  for (std::multimap<std::string, dataRelationValue_t>::iterator iter =
         myMap.lower_bound(dataID);
       iter != myMap.upper_bound(dataID);
       ++iter) {
    ++ix;
  }
  myLock.unlock();
  return ix;
} // getRelationNb

unsigned int
DataRelationMgr::getRelationMaxIndex(const std::string &dataID) {
  int max = 0;
  myLock.lock();
  for (std::multimap<std::string, dataRelationValue_t>::iterator iter =
         myMap.lower_bound(dataID);
       iter != myMap.upper_bound(dataID);
       ++iter) {
    if (iter->second.index > max) {
      max = (unsigned int) iter->second.index;
    }
  }
  myLock.unlock();
  return max;
} // getRelationMaxIndex

/**
 * Display all the relations for debug
 */
void
DataRelationMgr::displayContent() {
  if (TRACE_LEVEL >= TRACE_ALL_STEPS) {
    std::cout << "Content of relationship DB: \n";
    myLock.lock();
    std::multimap<std::string, dataRelationValue_t>::iterator iter =
      myMap.begin();
    for (; iter != myMap.end(); ++iter) {
      std::cout << " | " << iter->first
                << " | " << iter->second.ID
                << " | " << iter->second.index
                << " | " << iter->second.flag << "\n";
    }
    myLock.unlock();
  }
} // displayContent
