/**
 * @file client.c
 *
 * @brief  plugin_example, client side
 *
 * @author  Alan SU (Alan.Su@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */
/****************************************************************************/
/* plugin_example, client side                                              */
/*                                                                          */
/*   This example simulates a database search application, in which         */
/*   servers have access to a fixed set of databases, and clients           */
/*   submit queries against these databases.  The servers in this           */
/*   example make use of DIET's plugin scheduler facility to perform        */
/*   client request scheduling based on optimization of the                 */
/*   availability of the requested database(s) at the targeted              */
/*   server nodes.                                                          */
/****************************************************************************/


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "DIET_client.h"

/* forward declarations of private functions */
static char *
constructTargetString(int num, char **strings);

/*
** MAIN
*/
int
main(int argc, char **argv) {
  int retval;
  diet_profile_t *profile = NULL;
  char *targetString;

  if (argc < 3) {
    fprintf(stderr,
            "usage: %s <client config> <resource> [<resource> ...]\n",
            argv[0]);
    exit(1);
  }

  if ((retval = diet_initialize(argv[1], argc, argv)) != 0) {
    fprintf(stderr, "%s: DIET initialization failed (%d)\n", argv[0], retval);
    exit(1);
  }

  if ((profile = diet_profile_alloc("DBsearch", 0, 0, 0)) == NULL) {
    fprintf(stderr, "%s: unable to allocate DIET profile\n", argv[0]);
    exit(1);
  }

  /*
  ** create a colon-separated string representing the databases
  ** that the user has requested be queried
  */
  targetString = constructTargetString(argc - 2, &(argv[2]));
  if (diet_paramstring_set(diet_parameter(profile, 0),
                           targetString,
                           DIET_VOLATILE) != 0) {
    fprintf(stderr, "%s: unable to set target string\n", argv[0]);
    exit(1);
  }

  if ((retval = diet_call(profile)) != 0) {
    fprintf(stderr, "%s: DIET call failed (%d)\n", argv[0], retval);
    exit(1);
  }

  free(targetString);
  diet_profile_free(profile);
  diet_finalize();

  exit(0);
} /* main */

/*
** constructTargetString: create colon-separated string from an
**   argv-style series of elements
*/
static char *
constructTargetString(int num, char **strings) {
  char *s;
  int stringLengthCount = 0;
  int strIter;

  assert(num > 0);
  assert(strings != NULL);

  for (strIter = 0; strIter < num; strIter++) {
    stringLengthCount += strlen(strings[strIter]);
  }
  s = calloc(num + stringLengthCount, sizeof(char));
  assert(s != NULL);

  for (strIter = 0; strIter < num - 1; strIter++) {
    sprintf(s + strlen(s), "%s:", strings[strIter]);
  }
  sprintf(s + strlen(s), "%s", strings[num - 1]);

  return (s);
} /* constructTargetString */
