/**
 * @file  est_internal.cc
 *
 * @brief  Plug-in scheduler: DIET-internal estimation vector access functions
 *
 * @author  Alan Su (Alan.Su@ens-lyon.fr)
 *
 * @section Licence
 *
 * Copyright Inria, ENS Lyon and UCBL (2000-2017) 
 * Copyright SysFera (2010-2015)
 *
 * - Eddy.Caron@ens-lyon.fr (Project Manager)
 *
 * This software is a computer program whose purpose is to provide an
 * easy and transparent access to distributed and heterogeneous
 * platforms.
 *
 *
 * This software is governed by the CeCILL license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided
 * only with a limited warranty  and the software's author,  the holder
 * of the economic rights,  and the successive licensors  have only
 * limited liability.
 *
 * In this respect, the user's attention is drawn to the risks
 * associated with loading,  using,  modifying and/or developing or
 * reproducing the software by the user in light of its specific status
 * of free software, that may mean  that it is complicated to
 * manipulate, and  that  also therefore means  that it is reserved for
 * developers and experienced professionals having in-depth computer
 * knowledge. Users are therefore encouraged to load and test the
 * software's suitability as regards their requirements in conditions
 * enabling the security of their systems and/or data to be ensured and,
 * more generally, to use and operate it in the same conditions as
 * regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL license and that you accept its terms.
 *
 */

#ifndef OLD_EV
#include "DIET_data.h"

#include "debug.hh"
#include "est_internal.hh"

#define EV_ELT(e, i) (((e)->estValues)[(i)])
#define EV_LEN(e) ((e)->estValues.length())
#define EV_SETLEN(e, l) ((e)->estValues.length(l))

int
diet_est_set_internal(estVector_t ev, int tag, double value) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", -1);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": tag must be non-negative (" << tag << ")\n", -1);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == (EV_ELT(ev, valIter)).v_tag &&
        (EV_ELT(ev, valIter)).v_idx < 0) {
      (EV_ELT(ev, valIter)).v_value.d(value);
      return (1);
    }
  }

  EV_SETLEN(ev, EV_LEN(ev) + 1);
  (EV_ELT(ev, EV_LEN(ev) - 1)).v_tag = tag;
  (EV_ELT(ev, EV_LEN(ev) - 1)).v_idx = EST_IDX_SCALAR;
  (EV_ELT(ev, EV_LEN(ev) - 1)).v_value.d(value);

  return (1);
} // diet_est_set_internal

double
diet_est_get_internal(estVectorConst_t ev, int tag, double errVal) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", errVal);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ <<
          ": tag must be non-negative (" <<
          tag <<
          ")\n",
          errVal);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == (EV_ELT(ev, valIter)).v_tag &&
        (EV_ELT(ev, valIter)).v_idx < 0) {
      return ((EV_ELT(ev, valIter)).v_value.d());
    }
  }

  return (errVal);
} // diet_est_get_internal

int
diet_est_defined_internal(estVectorConst_t ev, int tag) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", -1);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": tag must be non-negative (" << tag << ")\n", -1);
  }

  // cout << "AS: [" << __FUNCTION__ << "] num values = " << EV_LEN(ev) << "\n";

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == (EV_ELT(ev, valIter)).v_tag &&
        (EV_ELT(ev, valIter)).v_idx < 0) {
      return (1);
    }
  }

  return (0);
} // diet_est_defined_internal

int
diet_est_array_size_internal(estVectorConst_t ev, int tag) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", -1);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": tag must be non-negative (" << tag << ")\n", -1);
  }

  int maxIdx = -1;
  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == (EV_ELT(ev, valIter)).v_tag &&
        maxIdx < (EV_ELT(ev, valIter)).v_idx) {
      maxIdx = (EV_ELT(ev, valIter)).v_idx;
    }
  }

  return (maxIdx + 1);
} // diet_est_array_size_internal

int
diet_est_array_set_internal(estVector_t ev, int tag, int idx, double value) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", -1);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": tag must be non-negative (" << tag << ")\n", -1);
  }
  if (idx < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": idx must be non-negative (" << idx << ")\n", -1);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == (EV_ELT(ev, valIter)).v_tag &&
        (EV_ELT(ev, valIter)).v_idx == idx) {
      (EV_ELT(ev, valIter)).v_value.d(value);
      return (1);
    }
  }

  EV_SETLEN(ev, EV_LEN(ev) + 1);
  (EV_ELT(ev, EV_LEN(ev) - 1)).v_tag = tag;
  (EV_ELT(ev, EV_LEN(ev) - 1)).v_idx = idx;
  (EV_ELT(ev, EV_LEN(ev) - 1)).v_value.d(value);

  return (1);
} // diet_est_array_set_internal

double
diet_est_array_get_internal(estVectorConst_t ev,
                            int tag,
                            int idx,
                            double errVal) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", errVal);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ <<
          ": tag must be non-negative (" <<
          tag <<
          ")\n",
          errVal);
  }
  if (idx < 0) {
    ERROR_DEBUG(__FUNCTION__ <<
          ": idx must be non-negative (" <<
          idx <<
          ")\n",
          errVal);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == (EV_ELT(ev, valIter)).v_tag &&
        (EV_ELT(ev, valIter)).v_idx == idx) {
      return ((EV_ELT(ev, valIter)).v_value.d());
    }
  }

  return (errVal);
} // diet_est_array_get_internal

int
diet_est_array_defined_internal(estVectorConst_t ev, int tag, int idx) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", -1);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": tag must be non-negative (" << tag << ")\n", -1);
  }
  if (idx < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": idx must be non-negative (" << idx << ")\n", -1);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == (EV_ELT(ev, valIter)).v_tag &&
        (EV_ELT(ev, valIter)).v_idx == idx) {
      return (1);
    }
  }

  return (0);
} // diet_est_array_defined_internal

int
diet_est_set_str_internal(estVector_t ev, int tag, const char *value) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", -1);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": tag must be non-negative (" << tag << ")\n", -1);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == EV_ELT(ev, valIter).v_tag &&
        EV_ELT(ev, valIter).v_idx < 0) {
      EV_ELT(ev, valIter).v_value.s(value);
      return (1);
    }
  }

  EV_SETLEN(ev, EV_LEN(ev) + 1);
  EV_ELT(ev, EV_LEN(ev) - 1).v_tag = tag;
  EV_ELT(ev, EV_LEN(ev) - 1).v_idx = EST_IDX_SCALAR;
  EV_ELT(ev, EV_LEN(ev) - 1).v_value.s(value);

  return (1);
}

const char*
diet_est_get_str_internal(estVectorConst_t ev, int tag) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", NULL);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ <<
          ": tag must be non-negative (" <<
          tag <<
          ")\n",
          NULL);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == EV_ELT(ev, valIter).v_tag &&
        EV_ELT(ev, valIter).v_idx < 0) {
      return (EV_ELT(ev, valIter).v_value.s());
    }
  }

  return (NULL);
}

int
diet_est_set_bin_internal(estVector_t ev, int tag, const unsigned char *buf, size_t size) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", -1);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ << ": tag must be non-negative (" << tag << ")\n", -1);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == EV_ELT(ev, valIter).v_tag &&
        EV_ELT(ev, valIter).v_idx < 0) {
      SeqOctet b;
      b.length(size);
      memcpy(b.get_buffer(), buf, size);
      EV_ELT(ev, valIter).v_value.b(b);
      return (1);
    }
  }

  EV_SETLEN(ev, EV_LEN(ev) + 1);
  EV_ELT(ev, EV_LEN(ev) - 1).v_tag = tag;
  EV_ELT(ev, EV_LEN(ev) - 1).v_idx = EST_IDX_SCALAR;
  SeqOctet b;
  b.length(size);
  memcpy(b.get_buffer(), buf, size);
  EV_ELT(ev, EV_LEN(ev) - 1).v_value.b(b);

  return (1);
}

int
diet_est_get_bin_internal(estVectorConst_t ev, int tag, const unsigned char **buf, size_t *size) {
  if (ev == NULL) {
    ERROR_DEBUG(__FUNCTION__ << ": NULL estimation vector", -1);
  }
  if (tag < 0) {
    ERROR_DEBUG(__FUNCTION__ <<
          ": tag must be non-negative (" <<
          tag <<
          ")\n",
          -1);
  }

  for (unsigned int valIter = 0; valIter < EV_LEN(ev); valIter++) {
    if (tag == EV_ELT(ev, valIter).v_tag &&
        EV_ELT(ev, valIter).v_idx < 0) {
      *size = EV_ELT(ev, valIter).v_value.b().length();
      *buf = EV_ELT(ev, valIter).v_value.b().get_buffer();
      return (1);
    }
  }

  return (-1);
}

#endif /* OLD_EV */
